<?php

namespace App\Traits\PermissionManager;

use App\Models\Building;
use App\Models\PermissionManager\Permission;
use Spatie\Permission\Traits\HasRoles as SpatieHasRoles;

/**
 * Trait HasRoles
 *
 * @property \Illuminate\Database\Eloquent\Collection $roles
 * @property \Illuminate\Database\Eloquent\Collection $permissions
 *
 * @package App\Traits\PermissionManager
 *
 * @mixin \Illuminate\Database\Eloquent\Model
 */
trait HasRoles
{

    use SpatieHasRoles {
        SpatieHasRoles::roles as spatieRoles;
        SpatieHasRoles::permissions as spatiePermissions;
    }


    /**
     * A user may have multiple direct permissions.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function permissions()
    {
        return $this
            ->spatiePermissions()
            ->wherePivot('object_id', Building::getFromCookie()->id);
    }

    /**
     * A user may have multiple roles.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function roles()
    {
        return $this
            ->spatieRoles()
            ->wherePivot('object_id', Building::getFromCookie()->id);
    }

    /**
     * @param int $objectId
     * @param string|\App\Models\PermissionManager\Permission $permission
     *
     * @return bool
     * @throws \Spatie\Permission\Exceptions\PermissionDoesNotExist
     */
    public function hasObjectPermissionTo($objectId, $permission)
    {
        if (is_string($permission)) {
            $permission = app(Permission::class)->findByName($permission);
        }

        return
            $this->hasObjectDirectPermission($objectId, $permission)
            ||
            $this->hasObjectPermissionViaRole($objectId, $permission);
    }

    /**
     * @param int $objectId
     * @param string|\App\Models\PermissionManager\Permission $permission
     *
     * @return bool
     * @throws \Spatie\Permission\Exceptions\PermissionDoesNotExist
     */
    public function hasObjectDirectPermission($objectId, $permission)
    {
        if (is_string($permission)) {
            $permission = app(Permission::class)->findByName($permission);

            if (!$permission) {
                return false;
            }
        }

        return $this->spatiePermissions()
            ->wherePivot('object_id', $objectId)
            ->wherePivot('permission_id', $permission->id)->count() > 0;
    }

    /**
     * @param int $objectId
     * @param string|\App\Models\PermissionManager\Permission $permission
     *
     * @return bool
     * @throws \Spatie\Permission\Exceptions\PermissionDoesNotExist
     */
    protected function hasObjectPermissionViaRole($objectId, $permission)
    {
        if (is_string($permission)) {
            $permission = app(Permission::class)->findByName($permission);

            if (!$permission) {
                return false;
            }
        }

        return $this->hasRole($permission->roles);
    }

    /**
     * Determine if the user may perform the given permission.
     *
     * @param string|\App\Models\PermissionManager\Permission $permission
     *
     * @return bool
     *
     * @throws \Spatie\Permission\Exceptions\PermissionDoesNotExist
     */
    public function hasPermissionTo($permission)
    {
        // Denied access by default
        $hasPermissionTo = false;

        // Try to get provided permission
        $permission = self::getPermissionByName($permission);

        if (
            // If permission not found
            !$permission
            ||
            // Or if permission is not "full access"
            ($permission->name != Permission::FULL_ACCESS)
        ) {
            // Check for "full access" permission first
            $hasPermissionTo = $this->hasPermissionTo(Permission::FULL_ACCESS);
        }

        if (!$hasPermissionTo && $permission) {
            return
                $this->hasDirectPermission($permission)
                ||
                $this->hasPermissionViaRole($permission);
        }

        return $hasPermissionTo;
    }

    /**
     * @param string|\App\Models\PermissionManager\Permission $permission
     *
     * @return null|\App\Models\PermissionManager\Permission $permission
     * @throws \Spatie\Permission\Exceptions\PermissionDoesNotExist
     */
    protected static function getPermissionByName($permission)
    {
        if (is_string($permission)) {
            if ($permission = app(Permission::class)->findByName($permission)) {
                return $permission;
            }
        } elseif ($permission instanceof Permission) {
            return $permission;
        }

        return null;
    }

    /**
     * @return int
     */
    public function getMaxPermissionWeight()
    {
        // Get max permission weight for current user
        $maxPermissionWeight = (int)$this->permissions->max('weight');

        /** @var \Illuminate\Database\Eloquent\Collection $userRoles */
        $userRoles = $this->roles;
        if ($userRoles->count() > 0) {
            /** @var \App\Models\PermissionManager\Role $userRole */
            foreach ($userRoles as $userRole) {
                $maxRolePermissionWeight = (int)$userRole->permissions()->max('weight');
                if ($maxRolePermissionWeight > $maxPermissionWeight) {
                    $maxPermissionWeight = $maxRolePermissionWeight;
                }
            }
        }

        return $maxPermissionWeight;
    }

    /**
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getSubordinateByPermissions()
    {
        $maxPermissionWeight = $this->getAllPermissionsWeight();

        return static::where('id', '<>', $this->id)->get()->filter(function (self $value) use ($maxPermissionWeight) {
            return $value->getAllPermissionsWeight() <= $maxPermissionWeight;
        });
    }

    /**
     * Grant the given permission(s) to a role.
     *
     * @param string|array|\App\Models\PermissionManager\Permission|\Illuminate\Support\Collection ...$permissions
     *
     * @return $this
     */
    public function givePermissionTo(...$permissions)
    {
        $permissions = collect($permissions)
            ->flatten()
            ->mapWithKeys(function ($permission) {
                $permission = $this->getStoredPermission($permission);

                return [$permission->id => ['object_id' => Building::getFromCookie()->id]];
            })
            ->all();

        $this->permissions()->attach($permissions);

        $this->forgetCachedPermissions();

        return $this;
    }

    /**
     * Return all  permissions the directory coupled to the user.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getDirectPermissionsInAnyObject()
    {
        return $this->spatiePermissions()->withPivot('object_id')->get();
    }

    /**
     * Return all the permissions the user has via roles.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getPermissionsViaRolesInAnyObject()
    {
        return $this->load('roles', 'roles.permissions')->spatieRoles()->withPivot('object_id')->get()
            ->flatMap(function ($role) {
                return $role->permissions->map(function ($permission) use ($role) {
                    $permission->pivot->object_id = $role->pivot->object_id;
                    return $permission;
                });
            })->values();
    }

    /**
     * Return all the permissions the user has, both directly and via roles.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getAllPermissionsInAnyObject()
    {
        return $this->getDirectPermissionsInAnyObject()->concat($this->getPermissionsViaRolesInAnyObject())->values();
    }

    public function getAccessableObjectsByPermissions()
    {
        return $this->getAllPermissionsInAnyObject()
            ->map(function (Permission $permission) {
                return $permission->pivot->object_id;
            })->unique();
    }

    /**
     * Determine if the user has any of the given permissions.
     *
     * @return bool
     */
    public function hasAnyPermissionInAnyObject()
    {
        if ($this->getDirectPermissionsInAnyObject()->count() > 0) {
            return true;
        }

        if ($this->getPermissionsViaRolesInAnyObject()->count() > 0) {
            return true;
        }

        return false;
    }

    /**
     * Returns sum of weights of all permissions in current object
     *
     * @return int
     */
    public function getAllPermissionsWeight()
    {
        return (int)$this->getAllPermissions()->sum('weight');
    }
}
