<?php
namespace App\Traits\PermissionManager;

use Illuminate\Support\Facades\Auth;

/**
 * Trait CheckingPermissions.
 *
 * Should be used only in controllers, which inherited from \Backpack\CRUD\app\Http\Controllers\CrudController.
 * Usage:
 *
 * <?php
 *
 * namespace App\Http\Controllers\Admin;
 *
 * use Backpack\CRUD\app\Http\Controllers\CrudController;
 *
 * class ExampleController extends CrudController
 * {
 *     use CheckingPermissions;
 *
 *     public function __construct()
 *     {
 *         parent::__construct();
 *
 *         $this->applyCheckingPermissions([[
 *             [Permission::APARTMENTS_CREATE_ACCESS],
 *             ['create', 'store'],
 *         ], [
 *             [Permission::APARTMENTS_VIEW_FULL_ACCESS],
 *             ['show'],
 *         ], [
 *             [Permission::APARTMENTS_VIEW_FULL_ACCESS, Permission::APARTMENTS_VIEW_LIMITED_ACCESS],
 *             ['index'],
 *         ], [
 *             [Permission::APARTMENTS_EDIT_ACCESS],
 *             ['edit', 'update'],
 *         ], [
 *             [Permission::FULL_ACCESS],
 *             ['resetActivations'],
 *         ], [
 *             [Permission::APARTMENTS_DELETE_ACCESS],
 *             ['destroy']
 *         ]]);
 *     }
 *
 *     public function setup()
 *     {
 *         $this->applyBackpackPermissions();
 *     }
 * }
 *
 *
 * @package App\Traits\PermissionManager
 *
 * @mixin \Backpack\CRUD\app\Http\Controllers\CrudController
 */
trait CheckingPermissions
{

    /**
     * Storage for permissions rules
     *
     * @var array
     */
    protected $laravelPermissions = [];

    /**
     * Map for converting laravel-named permissions to backpack-named permissions.
     * For example, single permission for accessing entities list page called "index" in laravel,
     * but called "list" in backpack
     *
     * @var array
     */
    protected $laravelBackpackPermissionsMap = [
        'create'      => 'create',
        'index'       => 'list',
        'show'        => 'show',
        'update'      => 'update',
        'destroy'     => 'delete',
        'reorder'     => 'reorder',
        'revisions'   => 'revisions',
        'details_row' => 'details_row',
    ];


    /**
     * Sets and applies checking laravel's permissions to accessing actions and routes of controller.
     * Should be called from constructor.
     *
     * @param \Illuminate\Support\Collection|array $permissions
     */
    public function applyCheckingPermissions($permissions)
    {
        // Save provided rules for \App\Traits\PermissionManager\CheckingPermissions::applyBackpackPermissions() method needs
        $this->laravelPermissions = $permissions;

        // All backpack actions are disabled by default
        $this->crud->denyAccess($this->laravelBackpackPermissionsMap);

        if (is_array($permissions)) {
            $permissions = collect($permissions);
        }

        $permissions->each(function($permission) {
            // Apply laravel permissions via middlewares
            $this
                ->middleware('permission:' . implode('|', $permission[0]))
                ->only($permission[1]);
        });
    }

    /**
     * Applies checking backpack's permissions to accessing actions and routes of controller.
     * Should be called from setup() method of controller.
     */
    public function applyBackpackPermissions()
    {
        // Get current user
        $user = Auth::user();

        collect($this->laravelPermissions)->each(function ($permission) use ($user) {
            $laravelPermissionsNames = $permission[0];
            // Apply backpack permissions
            collect($permission[1])->each(function ($laravelPermissionName) use ($laravelPermissionsNames, $user) {
                if (
                    // According backpack's permission founded
                    !empty($this->laravelBackpackPermissionsMap[$laravelPermissionName])
                    &&
                    // And action is allowed by laravel's permission
                    $user->hasAnyPermission(...$laravelPermissionsNames)
                ) {
                    // Allow access to backpack's action too
                    $this->crud->allowAccess($this->laravelBackpackPermissionsMap[$laravelPermissionName]);
                }
            });
        });
    }
}
