<?php
namespace App\Traits\Admin;

use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;

trait Ordering {
    /**
     * Получение настроек для сортировки
     *
     * @return array
     */
    private function getOrderSettings()
    {
        $offset = $this->request->start;
        $limit = $this->request->length;
        $direction = $this->request->order[0]['dir'];

        return [$offset, $limit, $direction];
    }

    /**
     * Валидация настроек сортировки
     *
     * @param null $offset
     * @param null $limit
     * @param null $direction
     * @return \Illuminate\Support\Collection
     */
    private function validateOrderSettings($offset = null, $limit = null, $direction = null)
    {
        $status = collect([
            'offset'        => false,
            'limit'         => false,
            'direction'     => false
        ]);

        // Проверка входящиз значений limit, offset и direction
        $int_reg = '{^[0-9]+$}';
        if (preg_match($int_reg, $offset) && $offset > 0) {
            $status->put('offset', true);
        }

        if ((preg_match($int_reg, $limit) && $limit > 0)) {
            $status->put('limit', true);
        }

        // Проверка входящего значения direction
        if ((preg_match('{^asc|desc$}', $direction))) {
            $status->put('direction', true);
        }

        return $status;
    }

    /**
     * Сортировка по простым полям бд
     *
     * @param $row
     * @return \Illuminate\Support\Collection
     */
    protected function sortBySimpleRow($table, $row, $settings = [])
    {
        // Установки
        $settings['select'] = isset($settings['select'])? $settings['select'] : null;
        $settings['object'] = isset($settings['object'])? $settings['object'] : true;
        $settings['clause'] = isset($settings['clause'])? $settings['clause'] : null;
        $settings['type']   = isset($settings['type'])? $settings['type']     : 'natural';
        $settings['get']    = isset($settings['get'])? $settings['get']       : false;

        // Построитель запросов без сортировки
        $raw_object_builder = $this->createBuilder($table, $settings['select'], $settings['object'], $settings['clause']);

        // Настройки сортировки
        list($offset, $limit, $direction) = $this->getOrderSettings();
        $type = is_null($settings['type'])? 'natural' : $settings['type'];

        // Проверка входящих значений limit и offset
        $valid_settings = $this->validateOrderSettings($offset, $limit, $direction);

        if ($valid_settings['offset']) {
            $raw_object_builder->offset($offset);
        }

        if ($valid_settings['limit']) {
            $raw_object_builder->limit($limit);
        }

        if (!$valid_settings['direction']) {
            $direction = 'acs';
        }

        switch ($type) {
            case 'length':
                // Сортировка по длине
                $raw_object_builder->orderByRaw('
                    LENGTH('.$row.') '.$direction.',
                    LOWER('.$row.')+0 '.$direction.',
                    LOWER('.$row.')'.$direction
                );
                break;
            case 'classic':
                // Приоритет отдается "стандартной" сортировке
                $raw_object_builder->orderByRaw('
                    LOWER('.$row.') '.$direction.',
                    LOWER('.$row.')+0 '.$direction.',
                    LENGTH('.$row.') '.$direction
                );
                break;
            default:
                // "Натуральная" сортировка
                $raw_object_builder->orderByRaw('
                    LOWER('.$row.')+0 '.$direction.',
                    LOWER('.$row.') '.$direction.',
                    LENGTH('.$row.') '.$direction
                );
                break;
        }

        // Если необходимо вернуть весь массив для работы с функцией createOrderedObjects
        if ($settings['get']) {
            return $raw_object_builder->get();
        } else {
            return $raw_object_builder->pluck('id');
        }
    }

    /**
     * Создание объектов из отсортированного массива
     * Функция работает только для простых полей (которые не возвращают данные из смежных таблиц)
     * Для остальных случаев применяется createOrderedObjectsByIds
     *
     * @param $raw_claims
     * @return \Illuminate\Support\Collection
     */
    private function createOrderedObjects($model, $raw_objects)
    {
        // На основе полученных данных создать объекты
        $objects = collect();
        foreach ($raw_objects as $raw_object) {
            $object = new $model();
            foreach ($raw_object as $attribute => $value) {

                $object->$attribute = $value;
            }

            $objects->push($object);
        }

        return $objects;
    }

    /**
     * Создание объектов из отсортированного массива
     *
     * @param $raw_claims
     * @return \Illuminate\Support\Collection
     */
    private function createOrderedObjectsByIds($model, $ids)
    {
        // На основе полученных данных создать объекты
        $objects = collect();

        foreach ($ids as $id) {
            $objects->push($model::find($id));
        }

        return $objects;
    }

    /**
     * Создание "построителя" запросов. Вынесен в отдельный метод для возможности переопределения
     *
     * @param $table
     * @param null $selectRaw
     * @param bool $object
     * @param null $clause
     * @return \Illuminate\Database\Query\Builder
     */
    protected function createBuilder($table, $selectRaw = null, $object = true, $clause = null)
    {
        $select = is_null($selectRaw)? '*' : $selectRaw;

        // Получить все объекты из таблицы объектов
        $raw_object_builder = DB::table($table)
            ->selectRaw($select);

        if ($object) {
            $raw_object_builder->where('object_id', config('app.object.id'));
        }

        if ($clause) {
            $raw_object_builder->whereRaw($clause);
        }
        //dd($raw_object_builder->get());
        return $raw_object_builder;
    }
}
