<?php

namespace App\Providers;

use App\Models\ApartmentSipNumber;
use App\Models\UserPermissionsWeight;
use Backpack\Settings\SettingsServiceProvider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class AppServiceProvider extends ServiceProvider {

    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot(Request $request) {
        URL::forceScheme('https');
        Request::setTrustedProxies([$request->getClientIp()], Request::getTrustedHeaderSet());
        $this->app['request']->server->set('HTTPS', TRUE);


        Validator::extend('isexists', function ($attribute, $value, $parameters, $validator) {
            $element = DB::table($parameters[0])
                ->select($parameters[1])
                ->where($parameters[1], $value)
                ->get();
            return count($element) > 0 ? TRUE : FALSE;
        });
        Validator::extend('phone', function ($attribute, $value, $parameters, $validator) {
            $isValid = preg_match('/^((8|\+7)[\- ]?)?(\(?\d{3}\)?[\- ]?)?[\d\- ]{7,10}$/', $value);
            return $isValid;
        });
        Validator::extend('unique_sip', function($attribute, $value, $parameters, $validator) {

            $sip_numbers = collect(json_decode($value))->filter(function ($value) {
                if (isset($value->sip_number) || isset($value->sip_password)) {
                    return true;
                }

                return false;
            });

            $errors = collect();

            /*
             * Проверка на уникальные поля на этапе ввода
             * sns_u - дубликат массива sip_numbers для работы с ним
             * sn_u - sip номер из массива sip номеров
             * sns_coin - совпадающие sip номера
             * sn_cur - текущий проверяемый sip номер
             * $sn_wp - sip номера без пароля
             * $sns_long - список длинных sip номеров
             */

            $empty_rows = false;

            $sns_u = $sip_numbers->map(function ($value) use (&$empty_rows) {
                if (isset($value->sip_number)) {
                    return $value->sip_number;
                } else {
                    $empty_rows = true;
                }
            });

            if ($empty_rows) {
                $errors->push(trans('apartments.validation.sip_numbers_empty_rows'));
            }

            // Проверка на наличия пароля у sip номеров
            $sn_wp = collect();
            foreach ($sip_numbers as $sn) {
                if (empty($sn->sip_password)) {
                    if (!empty($sn->sip_number)) {
                        $sn_wp->push($sn->sip_number);
                    }
                }
            } unset($sn);

            if ($sn_wp->count()) {
                // Уловие для "плюрализации" (один sip номер, два sip номера)
                if ($sn_wp->count() === 1) {
                    $errors->push(trans('apartments.validation.sip_password_empty', [
                        'number' => $sn_wp->first()
                    ]));
                } else {
                    $errors->push(trans('apartments.validation.sip_passwords_empty', [
                        'numbers' => $sn_wp->implode(', ')
                    ]));
                }
            }

            // До тех пор, пока в проверочном массиве остались значения, выполнять проверку на уникальность
            $sns_coin = collect();
            while ($sns_u->count()) {
                $sn_cur = $sns_u->shift();

                // Так как sns_u - массив из объектов, выполняем полный проход по нему, вместо обычного поиска совпадающих элементов
                foreach ($sns_u as &$sn) {
                    // Если есть совпадение, добавляем его в соответствующий массив и удаляем из старого все подобные элементы
                    if ($sn_cur === $sn) {
                        // Проверка на пустые значения
                        if (!empty($sn_cur)) {
                            $sns_coin->push($sn_cur);
                        }

                        // Удалить все повторяющиеся sip номера
                        $sns_u = $sns_u->reject(function($value) use ($sn_cur) {
                            return $value === $sn_cur;
                        });
                        break;
                    }
                } unset($sn);
            }

            $sns_coin_limit = 30;
            if ($sns_coin->count()) {
                $sns_coin->transform(function ($value) use ($sns_coin_limit) {
                    if (Str::length($value) > $sns_coin_limit) {
                        return Str::limit($value, $sns_coin_limit);
                    }
                });

                // Уловие для "плюрализации" (один sip номер, два sip номера)
                if ($sns_coin->count() === 1) {
                    $errors->push(trans('apartments.validation.not_unique_sip_number', [
                        'number' => $sns_coin->first()
                    ]));
                } else {
                    $errors->push(trans('apartments.validation.not_unique_sip_numbers', [
                        'numbers' => $sns_coin->implode(', ')
                    ]));
                }
            }

            // Проверка на длину вводимых sip-номеров
            foreach ($sip_numbers as $sn) {
                if (isset($sn->sip_number) && mb_strlen($sn->sip_number) > 255) {
                    $errors->push(trans('apartments.validation.sip_numbers_is_to_long', [
                        'symbols' => 255
                    ]));
                    break;
                }
            } unset($sn);

            // Проверка на длину вводимых sip-паролей
            foreach ($sip_numbers as $sn) {
                if (isset($sn->sip_password) && mb_strlen($sn->sip_password) > 255) {
                    $errors->push(trans('apartments.validation.sip_passwords_is_to_long', [
                        'symbols' => 255
                    ]));
                    break;
                }
            } unset($sn);

            $not_unique_sip_numbers = [];

            // Get object sip numbers without existing in current apartment key
            $object_sip_numbers = ApartmentSipNumber::getSipNumbersWithObjectId($sip_numbers->pluck('id'))->pluck('sip_number')->toArray();

            foreach ($sip_numbers as $sip_number) {
                if (!isset($sip_number->sip_number)) {
                    continue;
                }

                $checking_sip_number = ApartmentSipNumber::getSipNumberWithObjectId($sip_number->sip_number);

                if ($checking_sip_number instanceof ApartmentSipNumber) {
                    if (!isset($sip_number->apartment_id)) {
                        $not_unique_sip_numbers[] = $sip_number->sip_number;
                        continue;
                    }
                    if ($checking_sip_number->sip_number != $sip_number->sip_number) {
                        $not_unique_sip_numbers[] = $sip_number->sip_number;
                    }

                    // Check existing sip numbers
                    if (isset($sip_number->id)) {
                        if (in_array($sip_number->sip_number, $object_sip_numbers)) {
                            $not_unique_sip_numbers[] = $sip_number->sip_number;
                        }
                    }
                }
            }

            $not_unique_sip_numbers = array_unique($not_unique_sip_numbers);

            if (!empty($not_unique_sip_numbers)) {
                // Уловие для "плюрализации" (один sip номер, два sip номера)
                if (count($not_unique_sip_numbers) === 1) {
                    $errors->push(trans('apartments.validation.sip_number_already_exists', [
                        'number' => array_shift($not_unique_sip_numbers)
                    ]));
                } else {
                    $errors->push(trans('apartments.validation.sip_numbers_already_exists', [
                        'numbers' => implode($not_unique_sip_numbers, ', ')
                    ]));
                }
            }

            if ($errors->count()) {
                $validator->setCustomMessages($errors->toArray());
                return false;
            }

            return true;
        });

        \App\Models\User::created(function($user) use($request) {
            $permissions_weight = $user->getPermissionsWeightByDataInRequest($request);
            $user_id = $user->id;
            $object_id = config('app.object.id');

            if(isset($permissions_weight) && !empty($user_id) && !empty($object_id)) {
                if(UserPermissionsWeight::where('user_id', $user_id)->where('object_id', $object_id)->exists()) {
                    $upw = UserPermissionsWeight::where('user_id', $user_id)->where('object_id', $object_id)->first();
                } else {
                    $upw = new UserPermissionsWeight;
                }

                $upw->user_id = $user_id;
                $upw->object_id = $object_id;
                $upw->weight = $permissions_weight;
                $upw->save();
            }
        });

        \App\Models\User::updated(function($user) use($request) {
            $permissions_weight = $user->getPermissionsWeightByDataInRequest($request);
            $user_id = $user->id;
            $object_id = config('app.object.id');

            if(isset($permissions_weight) && !empty($user_id) && !empty($object_id)) {
                if(UserPermissionsWeight::where('user_id', $user_id)->where('object_id', $object_id)->exists()) {
                    $upw = UserPermissionsWeight::where('user_id', $user_id)->where('object_id', $object_id)->first();
                } else {
                    $upw = new UserPermissionsWeight;
                }
                $upw->user_id = $user_id;
                $upw->object_id = $object_id;
                $upw->weight = $permissions_weight;
                $upw->save();
            }
        });

    }

    /**
     * Register any application services.
     *
     * @return void
     */
    public function register() {
        if ($this->checkDefaultDbConnection()) {
            // Register this service providers only if DB connection correct
            $this->app->register(SettingsServiceProvider::class);
        }

        if ($this->app->environment() !== 'production') {
            $this->app->register(\Barryvdh\LaravelIdeHelper\IdeHelperServiceProvider::class);
        }

        //
    }

    /**
     * @return bool
     */
    private function checkDefaultDbConnection()
    {
        try {
            DB::connection()->getPdo();
        } catch (\Exception $e) {
            return false;
        }

        return true;
    }
}
