<?php

namespace App\Models;

use App\Traits\UseClaimTypes;
use Backpack\CRUD\CrudTrait;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Laravel\Passport\HasApiTokens;
use Laravel\Passport\PersonalAccessTokenResult;
use App\Traits\PermissionManager\HasRoles;
use Illuminate\Notifications\Notifiable;
use Illuminate\Foundation\Auth\User as Authenticatable;
use App\Notifications\ResetPasswordNotification;
use App\Models\PermissionManager\Permission;

class User extends Authenticatable
{
    use CrudTrait;
    use HasRoles;
    use Notifiable;
    use HasApiTokens {
        createToken as traitCreateToken;
    }
    use UseClaimTypes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name', 'email', 'password', 'is_active', 'remember_marker'
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
        'permissions',
        'roles',
        'created_at',
        'updated_at',
        'pivot',
        'permissions_weight',
        'remember_marker'
    ];

    protected $appends = ['device', 'address'];

    public function devices()
    {
        return $this->hasManyThrough('App\Models\Device', 'Laravel\Passport\Token', 'user_id', 'id');
    }

    public function device()
    {
        $token = $this->token();
        return $token ? $token->belongsTo(Device::class, 'id') : null;
    }

    public function claimsTypes()
    {
        return $this->hasMany('App\Models\UserClaimType');
    }

    public function apartments()
    {
        return $this->belongsToMany(Apartment::class);
    }

    public function apartment()
    {
        $device = $this->device()->first();
        return $device ? Device::find($device->id)->apartment()->first() : null;
    }

    public function getDeviceAttribute()
    {
        $device = $this->device();
        return $device ? $device->get() : null;
    }

    /**
     * @return string
     */
    public function getAddressAttribute()
    {
        // this is default address from Config
        $address = Config::get('settings.house_address');

        // if user already authorized, then he have a device. Then we should check apartment's existence

        $device = $this->device()->first();

        if ($device &&  $device->apartment) {
            $apartment = $device->apartment;

            $address = $apartment->object->address;
            $address .= ', '.trans('user.attributes.apartment_reduction').' №' . $apartment->number;
        }

        return $address;
    }

    /**
     * Get push tokens by array of users id
     *
     * @param array $users
     * @param boolean $os_group
     *
     * @return \Illuminate\Support\Collection|static
     */
    public function getPushTokens(array $users, $os_group = false)
    {
        if(empty($users)) {
            $users = [0];
        }

        $devices = Device::select('push_token', 'is_ios')->where('push_token','!=', '')
            ->join('oauth_access_tokens', 'oauth_access_tokens.id', '=', 'devices.id')
            ->join('users', 'users.id', '=', 'oauth_access_tokens.user_id')
            ->whereIn('users.id', $users)
            ->get();

        if($os_group) {
            $devices = $devices->groupBy('is_ios');
        }

        return $devices;
    }

    public function getStyleType() {
        if ($style = $this->hasOne('App\Models\UserStyle', 'user_id', 'id')
            ->first()) {
            return Style::find($style->style_id)->type;
        }
        return config('style.default');
    }

    /**
     * Send the password reset notification.
     *
     * @param  string $token
     *
     * @return void
     */
    public function sendPasswordResetNotification($token) {
        $this->notify(new ResetPasswordNotification($token));
    }

    /**
     * Create a new personal access token for the user.
     * By default use current user permissions as scopes.
     *
     * @param  string  $name
     * @param  array  $scopes
     * @return \Laravel\Passport\PersonalAccessTokenResult
     */
    public function createToken($name, array $scopes = []): PersonalAccessTokenResult
    {
        if( ! $scopes) {
            $scopes = $this
              ->getAllPermissions()
              ->pluck('name')
              ->map(function ($val) {
                // scope should urlencoded for OAuth
                return urlencode($val);
              })
              ->toArray()
            ;
        }

        return $this->traitCreateToken($name, $scopes);
    }

    public function weights()
    {
        return $this->hasMany('App\Models\UserPermissionsWeight');
    }

    public function getPermissionsWeightByDataInRequest($request) {
        $all_permissions_weight = Permission::pluck('weight')->toArray();
        $max_weight =  max($all_permissions_weight);
        $weights = [0];

        if(!empty($request->get('permissions_show'))) {
            $permissions = Permission::whereIn('id', $request->get('permissions_show'))->get();
            if(!empty($permissions)) {
                foreach($permissions as $permission) {
                    $weights[$permission->id] = $permission->weight;
                }
            }
        }

        if(!empty($request->get('roles'))) {
            $roles = $request->get('roles');
            if(!empty($roles)) {
                $permissions_from_roles = Permission::whereHas('roles', function($q) use($roles) {
                    $q->where('id', '=', $roles);
                })->get();

            }

            if(!empty($permissions_from_roles)) {
                foreach($permissions_from_roles as $permission_from_roles) {
                    $weights[$permission_from_roles->id] = $permission_from_roles->weight;
                }
            }
        }

        if(array_sum($weights) > $max_weight){
            $weight = $max_weight;
        } else {
            $weight = array_sum($weights);
        }

        return $weight;
    }

    public function getPermissionsWeight() {
        $user_permissions = DB::select('SELECT permission_id FROM permission_users WHERE user_id = ? AND object_id = ?', [$this->id, config('app.object.id')]);
        $permission_ids = [];
        foreach ($user_permissions as $permission) {
            $permission_ids[] = $permission->permission_id;
        }

        $permissions = Permission::find($permission_ids);

        if ($permissions) {
            return $permissions->sum('weight');
        }

        return 0;
    }
}