<?php

namespace App\Models;

use Backpack\CRUD\CrudTrait;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class Advert extends Model {

    use CrudTrait;

    protected $hidden = ['created_at', 'updated_at', 'push_sent'];
    protected $fillable = [
        'object_id',
        'theme',
        'text',
        'actual_from_date',
        'actual_to_date',
    ];

    public function destinations()
    {
        return $this->hasMany(AdvertDestination::class);
    }

    public function viewed()
    {
        return $this->hasMany(AdvertViewed::class);
    }

    public function getActualTimeTo()
    {
        $actualToDate = $this->attributes['actual_to_date'];
        return "<span class='actual-date-to'>".$actualToDate."</span>";
    }

    public function getWhom()
    {
        $id = $this->id;
        return $this->getDestinations($id);
    }

    public function getDestinations($id)
    {
        $destinations = AdvertDestination::where('advert_id', $id)->get();
        if ($destinations[0]->is_all_house) {
            return __('adverts.sent_result.is_all_house');
        }
        $recipients = [];
        foreach ($destinations as $destination) {
            if ($destination->entrance) {
                $recipients[] = __('apartments.crud_fields.entrance').' '.$destination->entrance;
            }
            if ($destination->floor) {
                $recipients[] = __('apartments.crud_fields.floor').' '.$destination->floor;
            }
            if ($destination->apartment) {
                $recipients[] = __('apartments.apartment').' '.$destination->apartment;
            }
        }

        return implode(', ', $recipients);
    }

    public function getCountDestinations($id, $objectId)
    {
        $destinations = AdvertDestination::where('advert_id', $id)->get();
        if ($destinations[0]->is_all_house) {
            return Apartment::where('object_id', $objectId)->count();
        }
        $apartments = [];
        foreach ($destinations as $destination) {
            $entrance_apartments = [];
            $floor_apartments = [];
            $apartment = [];
            if ($destination->entrance) {
                $entrance_apartments = Apartment::where('entrance', $destination->entrance)->where('object_id', $objectId)->pluck('id')->toArray();
            }
            if ($destination->floor) {
                $floor_apartments = Apartment::where('floor', $destination->floor)->where('object_id', $objectId)->pluck('id')->toArray();
            }
            if ($destination->apartment) {
                $apartment = Apartment::where('number', $destination->apartment)->where('object_id', $objectId)->pluck('id')->toArray();
            }
            $apartments = array_merge($apartment, $entrance_apartments, $floor_apartments, $apartment);
        }
        return count(array_unique($apartments));
    }

    public function getViewed($id)
    {
        $users_id = AdvertViewed:: where('advert_id', $id)->pluck('user_id')->toArray();
        $apartments = [];
        foreach ($users_id as $user_id) {
            $apartment = User::find($user_id)->apartments()->pluck('number')->toArray();
            $apartments = array_merge($apartments, $apartment);
        }
        $apartments = array_sort($apartments);
        return $apartments;
    }

    /**
     * Является ли заявка действующей
     *
     * @return boolean
     */
    public function isActive()
    {
        $date = date('Y-m-d');

        if ($this->actual_from_date <= $date && $date <= $this->actual_to_date) {
            return true;
        }

        return false;
    }


    /**
     * Будет ли заявка действительна в будущем
     *
     * @return boolean
     */
    public function isWillActive()
    {
        $date = date('Y-m-d');

        if ($this->actual_from_date > $date && $this->actual_from_date < $this->actual_to_date) {
            return true;
        }

        return false;
    }

    /* Возвращает список пользователей которым предназначется */
    public function getDestinationUserList()
    {
        $users_list = [];

        $destinations = $this->destinations()->select('floor', 'apartment', 'entrance', 'is_all_house')->first()->toArray();

        if(!empty($destinations)) {
            $apartments_list = null;

            $apartments = Apartment::where('object_id', config('app.object.id'));

            if(isset($destinations['apartment'])) $apartments->where('number', $destinations['apartment']);

            if(isset($destinations['floor'])) $apartments->where('floor', $destinations['floor']);

            if(isset($destinations['entrance'])) $apartments->where('entrance', $destinations['entrance']);

            $apartments_list = $apartments->get();

            if(!empty($apartments_list)) {
                foreach ($apartments_list as $apartment) {
                    $users = $apartment->users()->get()->pluck('id')->toArray();
                    $users_list = array_merge($users_list, $users);
                }
            }
        };
        return array_unique($users_list);
    }
}
