<?php

namespace App\Http\Requests;

use App\Http\Requests\Request;
use App\Models\Apartment;

class ApartmentRequest extends \Backpack\CRUD\app\Http\Requests\CrudRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        // only allow updates if the user is logged in
        return \Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        // Except unique rule value for update requests
        $entity_id = $this->get('id');

        $rules = [
            'entrance' => 'required|max:255',
            'floor' => 'required|integer',
            'number' => 'required|regex:/^[1-9]+[0-9a-zа-я]*$/iu|max:255|unique_with:apartments,object_id,' .$entity_id,
            'sip_numbers' => 'unique_sip',
            'activation_limit' => 'required|integer'
        ];

        if ($entity_id > 0) {
            $rules['activation_limit'] =  'required|integer|min:' . (int)$this->getMinActivationLimit($entity_id);
        }

        return $rules;
    }

    /**
     * Get the validation attributes that apply to the request.
     *
     * @return array
     */
    public function attributes()
    {
        $attributes = ['number', 'entrance', 'floor', 'activation_code', 'object_id'];

        foreach ($attributes as $attribute_name) {
            $attributes[$attribute_name] = __("apartments.crud_fields.$attribute_name");
        }

        return $attributes;
    }

    /**
     * Get the validation messages that apply to the request.
     *
     * @return array
     */
    public function messages()
    {
        return [
            'unique_with' => __('apartments.validation.unique_with'),
            'activation_limit.min' => __('apartments.validation.activation_limit').' ('.__('apartments.validation.activation_limit_now').' :min)',
            'number.regex' => __('validation.regex') . ' ' . __('apartments.validation.number_example')
        ];
    }

    /**
     * Get minimum value of activations limit
     *
     * @return integer
     */
    protected function getMinActivationLimit(int $id)
    {
        $apartment = Apartment::find($id);

        return $apartment->getActivationsUsed();
    }
}
