<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Answer;
use App\Models\Review;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ReviewController extends Controller {

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     *
     * @return string
     */
    public function store(Request $request) {
        $user = Auth::user();
        $apartments = $user->apartments()->get();

        if ($apartments->count() === 0) {
            return response()->json(['message' => trans('reviews.validator.apartment_not_found')], 404);
        }

        if ($apartments->count() > 1) {
            return response()->json(['message' => trans('claims.validator.multiple_apartments')], 300);
        }

        $apartment = $apartments->first();

        Validator::make($request->all(), [
            'text' => 'required',
        ])->validate();

        $review = Review::create([
            'text' => $request->get('text'),
            'apartment' => $apartment->number,
            'object_id' => $apartment->object->id,
            'user_id' => $user->id,
        ]);

        Answer::create([
            'text' => $request->get('text'),
            'apartment' => $apartment->number,
            'review_id' => $review->id,
            'user_id' => $user->id,
            'is_concierge' => 0,
        ]);

        return '';
    }

    public function storeRecallAnswer(Request $request) {
        $user = Auth::user();
        $apartments = $user->apartments()->get();

        if ($apartments->count() === 0) {
            return response()->json(['message' => trans('reviews.validator.apartment_not_found')], 404);
        }

        if ($apartments->count() > 1) {
            return response()->json(['message' => trans('claims.validator.multiple_apartments')], 300);
        }

        $apartment = $apartments->first();

        Validator::make($request->all(), [
            'text' => 'required',
            'review_id' => 'required|numeric|isexists:reviews,id',
        ])->validate();

        $review_id = $request->get('review_id');

        Answer::create([
            'text' => $request->get('text'),
            'apartment' => $apartment->number,
            'review_id' => $review_id,
            'user_id' => $user->id,
        ]);

        $review = Review::find($review_id);
        $review->is_viewed = false;
        $review->save();

        return '';
    }

    public function getRecallAnswers($reviewId, Request $request) {
        $start = $request->get('start') ? : 0;
        $limit = $request->get('limit') ? : 100;
        $review = Review::findOrFail($reviewId);
        $answers = $review->answers();

        if ($this->answersAreEmpty($answers)) {
            $this->createAnswersFromReview($review);
            $answers = $review->answers();
        }
        $answers->offset($start);
        $answers->limit($limit);
        $answers = $answers->get();
        return $answers;
    }

    public function createAnswersFromReview($review)
    {
        Answer::create([
            'text' => $review->text,
            'apartment' => $review->apartment,
            'review_id' => $review->id,
            'user_id' => $review->user_id,
            'is_concierge' => 0,
        ]);
        if ($review->answer) {
            Answer::create([
                'text' => $review->answer,
                'apartment' => $review->apartment,
                'review_id' => $review->id,
                'user_id' => $review->user_id,
                'is_concierge' => 1,
            ]);
        }
    }

    public function answersAreEmpty($answers)
    {
        return $answers->count() === 0;
    }

    /**
     * Получение всех отывов пользователя
     */
    public function getRecall(Request $request) {
        $userId = \Auth::user()->id;
        return Review::where('user_id', $userId)->orderByDesc('id')
            ->withCount('answers')->get();
    }

}
