<?php

namespace App\Http\Controllers\Api;

use App\Models\QuizResult;
use Carbon\Carbon;
use App\Models\Quiz;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use App\Models\QuizDestination;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class QuizController extends Controller
{
    /**
     * Get quizzes list for user
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\JsonResponse | array
     */
    public function getUserQuizzes(Request $request)
    {
        $user = Auth::user();

        // Get User device: this is workaround for getting user's device.
        // See \App\Models\User::device() relation
        $device = $user->device();

        if ($device) {
            $device = $device->first();
        }

        if ($device && $device->apartment) {
            $objectId = $device->apartment->object->id;
            $userId = $user->id;
            $userApartments = $user->apartments;

            $apartments = (new QuizDestination)->where(function (Builder $query) use ($userApartments) {
                $query->whereIn('apartment', $userApartments->pluck('number'));
                foreach ($userApartments as $apartment) {
                    $query->orWhere(function (Builder $query) use ($apartment) {
                        $query->where('floor', '=', $apartment->floor);
                        $query->where('entrance', '=', $apartment->entrance);
                    });
                    $query->orWhere(function (Builder $query) use ($apartment) {
                        $query->where('floor', '=', null);
                        $query->where('entrance', '=', $apartment->entrance);
                    });
                    $query->orWhere(function (Builder $query) use ($apartment) {
                        $query->where('floor', '=', $apartment->floor);
                        $query->where('entrance', '=', null);
                    });
                }
                $query->orWhere('is_all_house', '=', 1);
            });

            $quizzesId = $apartments
                ->distinct(['quiz_destinations.quiz_id'])
                ->rightJoin('quizzes', 'quiz_destinations.quiz_id', '=', 'quizzes.id')
                ->where(function (Builder $query) use ($objectId) {
                    $query->where('quizzes.object_id', '=', $objectId);
                    $query->whereDate('quizzes.start_at', '<=', Carbon::now());
                    $query->whereDate('quizzes.end_at', '>=', Carbon::now());
                })
                ->get()
                ->pluck('quiz_id');

            return Quiz::with('answers','selectedAnswer', 'selectedFree')
                ->whereIn('quizzes.id', $quizzesId)
                ->select([
                    'quizzes.*',
                    \DB::raw('IF(`quiz_viewed`.`quiz_id` IS NULL, FALSE, TRUE) as `is_viewed`')
                ])
                ->distinct(['quizzes.id'])
                ->leftJoin('quiz_viewed', function ($join) use ($userId) {
                    $join
                        ->on('quiz_viewed.quiz_id', '=', 'quizzes.id')
                        ->on('quiz_viewed.user_id', '=', \DB::raw($userId));
                })
                ->orderBy('is_viewed')
                ->orderByDesc('start_at')
                ->get();
        }

        return [];
    }


    /**
     * @param int $quizId
     * @return int|mixed
     */
    public function markQuizAsRead($quizId)
    {
        try {
            $quiz = Quiz::findOrFail($quizId);
        } catch (\Exception $e) {
            return response()->json(['message' => trans('quizzes.quiz_not_found')], 404);
        }

        $data = [
            'user_id' => Auth::user()->id,
            'quiz_id' => $quiz->id,
        ];

        $isNotViewed = Validator::make($data, [
            'quiz_id' => 'unique_with:quiz_viewed,user_id',
        ])->passes();

        if ($isNotViewed) {
            $quiz->viewed()->create($data);
        }

        return $quiz;
    }


    /**
     * @param $id
     * @return \Illuminate\Database\Eloquent\Collection|\Illuminate\Database\Eloquent\Model
     */
    public function getQuizResults($id)
    {
        try {
            $quiz = Quiz::with(['answers' => function($query){
                $query->withCount('results');
            }])->select('id','title', 'question', 'is_private')->withCount('resultsFree')->findOrFail($id);
        } catch (\Exception $e) {
            return response()->json(['message' => trans('quizzes.quiz_not_found')], 404);
        }

        if ($quiz->is_private) {
            abort(403, trans('quizzes.results_are_private'));
        }
        return $quiz;
    }

    /**
     * @param int $quizId
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Contracts\Routing\ResponseFactory|\Symfony\Component\HttpFoundation\Response
     */
    public function setQuizAnswers($quizId, Request $request)
    {
        try {
            $quiz = Quiz::findOrFail($quizId);
        } catch (\Exception $e) {
            return response()->json(['message' => trans('quizzes.quiz_not_found')], 404);
        }

        $message = '';
        $issetResult = QuizResult::where('quiz_id', $quiz->id)
            ->where('user_id', Auth::user()->id)
            ->count();
        if($issetResult){
            $message = __('quizzes.result_isset');
        } else {
            if (!$quiz->multiple && count($request->answers) > 1) {
                $message = __('quizzes.multiple_is_not_allowed');
            } else {

                foreach ($request->answers as $answer) {
                    $validator = \Validator::make(['id' => $answer['id']], [
                        'id' => 'exists:quiz_answer,id',
                    ]);
                    if ($validator->fails()) {
                        $message = __('quizzes.answer_is_not_exists');
                    }
                }
            }
            if (!$quiz->free_answer && $request->free_answer) {
                $message = __('quizzes.free_is_not_allowed');
            }
        }

        if($message)
            return response()->json(['message' => $message],403);

        $result = new QuizResult();
        foreach ($request->answers as $answer) {
            $data = [
                'user_id' =>  Auth::user()->id,
                'quiz_id' => $quiz->id,
                'quiz_answer_id' => $answer['id']
            ];
            $result->create($data);
        }

        if($request->free_answer) {
            $data = [
                'user_id' =>  Auth::user()->id,
                'quiz_id' => $quiz->id,
                'free_answer_value' => $request->free_answer
            ];
            $result->create($data);
        }
    }
}
