<?php

namespace App\Http\Controllers\Api\Auth;

use App\Models\Apartment;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ActivationController extends LoginController
{

    /**
     * Get the login username to be used by the controller.
     *
     * @return string
     */
    public function username()
    {
        return 'number';
    }

    /**
     * Активация аккаунта по номеру квартиры и коду активации
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse | array
     */
    public function activation(Request $request)
    {
        $rules = [
          'number' => 'required_without:id|exists:apartments,number',
          'activation_code' => 'required',
          'id' => 'required_without:number'
        ];

        $this->validate($request, $rules);

        if ($request->get('number')) {
            $apartment = Apartment::where([
                'number' => $request->input('number'),
                'activation_code' => $request->input('activation_code'),
            ])->first();
        } else {
            $apartment = Apartment::where([
                'id' => $request->input('id'),
                'activation_code' => $request->input('activation_code'),
            ])->first();
        }

        // check an 'activation_code' related to apartment
        if ( ! $apartment) {
            $this->incrementLoginAttempts($request);
            $this->sendFailedLoginResponse($request);
        }

        $this->checkQuota($apartment);

        $user = $this->getOrCreateUser($apartment);

        Auth::login($user);

        return $this->sendLoginResponse($request, $apartment);

    }

    /**
     * API запросов от СКА: Выход из учетной записи от имени СКА
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function deleteToken()
    {
        // delete token and related device (by database constraint ON CASCADE)
        return response()->json([
            'success' => Auth::user()->token()->delete(),
        ], 200);
    }

    protected function checkQuota(Apartment $apartment)
    {
        if ( ! ($apartment->getActivationsUsed() < $apartment->activation_limit) ) {
            abort('409', __('apartments.activation_quota_exceeded'));
        }
    }

    protected function getOrCreateUser(Apartment $apartment)
    {
        if ($apartment->users()->exists()) {
            return $apartment->users()->first();
        } else {
            return $this->createUser($apartment);
        }
    }

    protected function createUser(Apartment $apartment)
    {
        $host = parse_url(url()->to('/'), PHP_URL_HOST);

        $user = User::create([
            'name' => __('auth.owner') . $apartment->number,
            'email'    => 'owner.' . $apartment->object->id. '-' . $apartment->number . '@' . $host,
            'password' => '',
        ]);

        $apartment->users()->save($user);

        return $user;
    }

}
