<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Advert;
use App\Models\AdvertDestination;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Config;

class AdvertController extends Controller {

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response | array
     */
    public function index() {
        $user = Auth::user();
        $userId = Auth::user()->id;
        $userApartments = Auth::user()->apartments()->get();
        $archiveTime = Config::get('settings.archival_time');
        $archiveTime = !empty($archiveTime) ? $archiveTime : 3;
        $archiveDate = Carbon::today()->subMonths($archiveTime);
        // Get User device: this is workaround for getting user's device.
        // See \App\Models\User::device() relation
        $device = $user->device();

        if ($device) {
            $device = $device->first();
        }

        if ($device && $device->apartment) {
            $objectId = $device->apartment->object->id;
            $userId = $user->id;
            $userApartments = $user->apartments;

            $apartments = AdvertDestination::where(function (Builder $query) use ($userApartments) {
                $query->whereIn('apartment', $userApartments->pluck('number'));
                foreach ($userApartments as $apartment) {
                    $query->orWhere(function (Builder $query) use ($apartment) {
                        $query->where('floor', '=', $apartment->floor);
                        $query->where('entrance', '=', $apartment->entrance);
                    });
                    $query->orWhere(function (Builder $query) use ($apartment) {
                        $query->where('floor', '=', null);
                        $query->where('entrance', '=', $apartment->entrance);
                    });
                    $query->orWhere(function (Builder $query) use ($apartment) {
                        $query->where('floor', '=', $apartment->floor);
                        $query->where('entrance', '=', null);
                    });
                }
                $query->orWhere('is_all_house', '=', 1);
            });

            $advertsId = $apartments
                ->distinct(['adverts_destinations.advert_id'])
                ->rightJoin('adverts', 'adverts_destinations.advert_id', '=', 'adverts.id')
                ->where(function (Builder $query) use ($objectId, $archiveDate) {
                    $query->where('adverts.object_id', '=', $objectId);
                    $query->whereDate('adverts.actual_to_date', '>=', $archiveDate);
                })
                ->get()
                ->pluck('advert_id');

            $advertsRaw = Advert::whereIn('id', $advertsId)
                    ->selectRaw('`id`, `theme`, `text`, `actual_from_date`, `actual_to_date`, IF(`adverts_viewed`.`advert_id` IS NULL, FALSE, TRUE) as `is_viewed`')
                    ->distinct(['adverts.id'])
                    ->leftJoin('adverts_viewed', function ($join) use ($userId) {
                        $join
                            ->on('adverts_viewed.advert_id', '=', 'adverts.id')
                            ->on('adverts_viewed.user_id', '=', \DB::raw($userId));
                    })
                    ->orderBy('is_viewed')
                    ->orderByDesc('actual_from_date')
                    ->orderByDesc('created_at')
                    ->get();
                    /*
             * Сортировка
             * - непрочитанные активные,
             * - прочитанные активные,
             * - непрочитанные архивные,
             * - прочитанные архивные.
             * */
            $advertsGroups = [[], [], [], []];
            foreach($advertsRaw as &$item) {
                if(Carbon::parse($item->actual_to_date) >= Carbon::now()) {
                    if(empty($item->is_viewed)) {
                        $advertsGroups[0][] = $item; // непрочитанные активные
                    }  else {
                        $advertsGroups[1][] = $item; // непрочитанные активные
                    }

                } else {
                    if(empty($item->is_viewed)) {
                        $advertsGroups[2][] = $item; // непрочитанные архивные,
                    }  else {
                        $advertsGroups[3][] = $item; // прочитанные архивные.
                    }
                }
            }

            $advertsSortRaw = [];

            foreach($advertsGroups as $key => $advertsGroup) {
                $advertsSortRaw = array_merge($advertsSortRaw, $advertsGroup);
            }

            return $advertsSortRaw;
        }

        return [];
    }

    /**
     * @param int $advertId
     *
     * @return \App\Models\Advert
     */
    public function markAsRead($advertId)
    {
        try {
            $advert = Advert::findOrFail($advertId);
        } catch (\Exception $e) {
            return response()->json(['message' => trans('adverts.advert_not_found')], 404);
        }

        $data = [
            'user_id' => Auth::user()->id,
            'advert_id' => $advert->id,
        ];

        $isNotViewed = Validator::make($data, [
            'advert_id' => 'unique_with:adverts_viewed,user_id',
        ])->passes();

        if ($isNotViewed) {
            $advert->viewed()->create($data);
        }

        $advert->is_viewed = 1;

        return $advert;
    }
}
