<?php

namespace App\Http\Controllers\Admin;

use App\Models\PermissionManager\Permission;
use App\Models\Style;
use App\Models\UserStyle;
use App\Traits\PermissionManager\CheckingPermissions;
use Illuminate\Support\Facades\Auth;
use App\Http\Requests\StyleRequest as StoreRequest;
use App\Http\Requests\StyleRequest as UpdateRequest;
use Backpack\CRUD\app\Http\Controllers\CrudController;

class UserStyleCrudController extends CrudController
{
    use CheckingPermissions;

    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::QUIZES_FULL_ACCESS, Permission::ANNOUNCEMENT_FULL_ACCESS],
            [
                'create',
                'store',
                'update',
            ],
        ]]);

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel("App\Models\UserStyle");
        $this->crud->setRoute(config('backpack.base.route_prefix', 'admin').'/templates/styles');
        $this->crud->setEntityNameStrings(__('styles.style_singular'), __('styles.style_plural'));

        /*
        |--------------------------------------------------------------------------
        | COLUMNS AND FIELDS
        |--------------------------------------------------------------------------
        */
        $this->crud->addField([
            'name' => 'style_id',
            'type' => 'hidden',
        ]);


        $this->crud->setCreateView('crud::style.create');
        $this->crud->setEditView('crud::style.edit');
        $this->crud->disableAutoFocus();

    }

    public function setup()
    {
        $this->applyBackpackPermissions();
    }

    public function index()
    {
        $this->crud->hasAccessOrFail('update');
        $this->crud->hasAccessOrFail('create');

        $user_id = Auth::User()->id;

        $this->crud->addField([
            'name' => 'user_id',
            'type' => 'hidden',
            'value' => $user_id
        ]);

        $this->data['crud'] = $this->crud;
        $this->data['title'] = ucfirst($this->crud->entity_name_plural);
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['entry'] = UserStyle::where('user_id', $user_id)->first();
        $this->data['styles'] = Style::get();

        if(!$this->data['entry']){
            return view($this->crud->getCreateView(), $this->data);
        } else {
            $this->data['id'] = $this->data['entry']->id;
            $this->data['fields'] = $this->crud->getUpdateFields($this->data['entry']->id);
            return view($this->crud->getEditView(), $this->data);
        }
    }

    public function store(StoreRequest $request) {
        $this->crud->hasAccessOrFail('create');

        // fallback to global request instance
        if (is_null($request)) {
            $request = \Request::instance();
        }

        // insert item in the db
        $item = $this->crud->create($request->except(['save_action', '_token', '_method']));
        $this->data['entry'] = $this->crud->entry = $item;

        // show a success message
        \Alert::success(trans('styles.success'))->flash();

        // save the redirect choice for next time
        $this->setSaveAction();

        return $this->performSaveAction($item->getKey());
    }

    public function update(UpdateRequest $request) {
        $this->crud->hasAccessOrFail('update');

        // fallback to global request instance
        if (is_null($request)) {
            $request = \Request::instance();
        }

        $style = UserStyle::where('user_id', $request->user_id)->first();
        $style->style_id = $request->style_id;
        $style->save();

        $this->data['entry'] = $this->crud->entry = $style;

        // show a success message
        \Alert::success(trans('styles.success'))->flash();
        return back();

    }
}
