<?php

namespace App\Http\Controllers\Admin;

use App\Models\Apartment;
use App\Models\Building;
use App\Models\PermissionManager\Permission;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\PermissionManager\app\Http\Controllers\UserCrudController as BackpackUserCrudController;
use Carbon\Carbon;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

/**
 * Class UserCrudController
 *
 * @package App\Http\Controllers\Admin
 */
class UserActivityController extends BackpackUserCrudController
{
    use CheckingPermissions;

    /**
     * @var string $permissionsRelation
     */
    protected $permissionsRelation = 'permissions';

    /**
     * @var string $rolesRelation
     */
    protected $rolesRelation = 'roles';


    /**
     * UserCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::FULL_ACCESS, Permission::AUTHORIZED_USERS_FULL_ACCESS, Permission::AUTHORIZED_USERS_LIMITED_ACCESS],
            ['index']
        ]]);
    }


    /**
     * Allow developers to set their configuration options for a CrudPanel.
     */
    public function setup()
    {
        $this->applyBackpackPermissions();
        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\Apartment');
        $this->crud->setEntityNameStrings(trans('backpack::permissionmanager.users_activity_report'), trans('backpack::permissionmanager.users_activity_report'));
        $this->crud->setRoute(config('backpack.base.route_prefix').'/user-activity');
        $this->crud->addClause('where', 'object_id', '=', config('app.object.id'));

        $this->crud->enableAjaxTable();

        $date = Carbon::createMidnightDate()->setTimezone('Europe/Moscow');
        // Columns.
        $this->crud->addColumn([
            'name' => 'entrance',
            'label' => __('apartments.crud_fields.entrance'),
            'type' => 'text',
        ]);

        $this->crud->addColumn([
            'name' => 'floor',
            'label' => __('apartments.crud_fields.floor'),
            'type' => 'text',
        ]);

        $this->crud->addColumn([
            'name' => 'number',
            'label' => __('apartments.crud_fields.number'),
            'type' => 'text',
        ]);

        $this->crud->addColumn([
            'name'  => 'last_activity',
            'label' => trans('active_users.crud.last_activity'),
            'type'  => 'model_function',
            'function_name' => 'getLastActivity',
        ]);

        $this->crud->addColumn([
            'label' => $date->subDays(7)
                    ->format('d.m') . ' - ' . $date->addDays(6)->format('d.m'),
            'type' => 'model_function',
            'function_name' => 'getActivityOneWeekAgo',
            'limit' => 100,
        ]);

        $this->crud->addColumn([
            'label' => $date->subDays(13)
                    ->format('d.m') . ' - ' . $date->addDays(6)->format('d.m'),
            'type' => 'model_function',
            'function_name' => 'getActivityTwoWeeksAgo',
            'limit' => 100,
        ]);

        $this->crud->addColumn([
            'label' => $date->subDays(13)
                    ->format('d.m') . ' - ' . $date->addDays(6)->format('d.m'),
            'type' => 'model_function',
            'function_name' => 'getActivityThreeWeeksAgo',
            'limit' => 100,
        ]);

        $this->crud->addColumn([
            'label' => $date->subDays(13)
                    ->format('d.m') . ' - ' . $date->addDays(6)->format('d.m'),
            'type' => 'model_function',
            'function_name' => 'getActivityFourWeeksAgo',
            'limit' => 100,
        ]);

        $this->crud->removeAllButtons();
    }


    /**
     * Display all rows in the database for this entity.
     *
     * @return Response
     * @throws \Backpack\CRUD\Exception\AccessDeniedException
     */
    public function index()
    {
        $this->crud->hasAccessOrFail('list');
        $this->data['crud'] = $this->crud;

        $this->crud->addButton('top', 'export', 'view', 'crud::buttons.export_to_excel', 'end');
        $this->data['title'] = __('active_users.crud.nav_string');
        $this->data['indexList'] = 'activity';

        // get all entries if AJAX is not enabled
        if (!$this->data['crud']->ajaxTable()) {
            $this->data['entries'] = $this->data['crud']->getEntries();
        }

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getListView(), $this->data);
    }

    public function show($id)
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle(trans('apartments.activity'));
        $apartments = Apartment::where('object_id', config('app.object.id'))->get();
        $object = Building::find(config('app.object.id'));
        $address = $object ? $object->address : "report";
        $sheet->setCellValue('A1', __('apartments.crud_fields.entrance'));
        $sheet->getColumnDimension('A')->setAutoSize(true);
        $sheet->setCellValue('B1', __('apartments.crud_fields.floor'));
        $sheet->getColumnDimension('B')->setAutoSize(true);
        $sheet->setCellValue('C1', __('apartments.crud_fields.number'));
        $sheet->getColumnDimension('C')->setAutoSize(true);
        $sheet->setCellValue('D1', trans('active_users.crud.last_activity'));
        $sheet->getColumnDimension('D')->setAutoSize(true);

        $date = Carbon::createMidnightDate()->setTimezone('Europe/Moscow');
        $sheet->setCellValue('E1', $date->subDays(7)
                ->format('d.m') . ' - ' . $date->addDays(6)->format('d.m'));
        $sheet->getColumnDimension('E')->setAutoSize(true);
        $sheet->setCellValue('F1', $date->subDays(13)
                ->format('d.m') . ' - ' . $date->addDays(6)->format('d.m'));
        $sheet->getColumnDimension('F')->setAutoSize(true);
        $sheet->setCellValue('G1', $date->subDays(13)
                ->format('d.m') . ' - ' . $date->addDays(6)->format('d.m'));
        $sheet->getColumnDimension('G')->setAutoSize(true);
        $sheet->setCellValue('H1', $date->subDays(13)
                ->format('d.m') . ' - ' . $date->addDays(6)->format('d.m'));
        $sheet->getColumnDimension('H')->setAutoSize(true);
        $sheet->getStyle('A1:H1')->getFont()->setBold(true);

        $i = 2;
        foreach ($apartments as $apartment) {
            $apartment->setExport();
            $sheet->setCellValue("A$i", $apartment->entrance);
            $sheet->getStyle("A$i")->getAlignment()
                ->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_LEFT);
            $sheet->setCellValue("B$i", $apartment->floor);
            $sheet->getStyle("B$i")->getAlignment()
                ->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_LEFT);
            $sheet->setCellValue("C$i", $apartment->number);
            $sheet->getStyle("C$i")->getAlignment()
                ->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_LEFT);
            $sheet->setCellValue("D$i", $apartment->getLastActivity());
            $sheet->getStyle("D$i")->getAlignment()
                ->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_LEFT);
            $sheet->getStyle("E$i")->getFill()
                ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                ->getStartColor()->setARGB($apartment->getActivityOneWeekAgo());
            $sheet->getStyle("F$i")->getFill()
                ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                ->getStartColor()->setARGB($apartment->getActivityTwoWeeksAgo());
            $sheet->getStyle("G$i")->getFill()
                ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                ->getStartColor()->setARGB($apartment->getActivityThreeWeeksAgo());
            $sheet->getStyle("H$i")->getFill()
                ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                ->getStartColor()->setARGB($apartment->getActivityFourWeeksAgo());
            $i++;
        }

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="'.$address.'.xlsx"');
        header('Cache-Control: max-age=0');

        $writer = new Xlsx($spreadsheet);
        $writer->save('php://output');
    }

}
