<?php

namespace App\Http\Controllers\Admin;

use App\Models\PermissionManager\Permission;
use App\Rules\DomainNameOrIp;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\Settings\app\Http\Controllers\SettingCrudController as BackpackSettingCrudController;
use Illuminate\Support\Facades\Auth;

// VALIDATION
use Backpack\Settings\app\Http\Requests\SettingRequest as UpdateRequest;
use Backpack\Settings\app\Models\Setting;
use Illuminate\Support\Facades\Validator;

class SettingCrudController extends BackpackSettingCrudController
{
    use CheckingPermissions;

    /**
     * RoleCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::SETTINGS_VIEW_ACCESS],
            ['index'],
        ], [
            [Permission::SETTINGS_EDIT_ACCESS],
            ['edit', 'update'],
        ]]);
    }

    public function setup()
    {
        parent::setup();

        $this->crud->setColumns([
            [
                'name'  => 'name',
                'label' => trans('backpack::settings.name'),
                'limit' => 500
            ],
            [
                'name'  => 'value',
                'label' => trans('backpack::settings.value'),
                'limit' => 500,
                'orderable' => false
            ],
            [
                'name'  => 'description',
                'label' => trans('backpack::settings.description'),
                'limit' => 500
            ],
        ]);

        $this->setButtons();

        $this->applyBackpackPermissions();
    }


    /**
     * Get the save configured save action or the one stored in a session variable.
     * @return [type] [description]
     */
    public function getSaveAction()
    {
        $saveAction = session('save_action', config('backpack.crud.default_save_action', 'save_and_back'));
        $saveOptions = [];
        $saveCurrent = [
            'value' => $saveAction,
            'label' => $this->getSaveActionButtonName($saveAction),
        ];

        switch ($saveAction) {
            case 'save_and_edit':
                $saveOptions['save_and_back'] = $this->getSaveActionButtonName('save_and_back');
                break;
            case 'save_and_back':
            default:
                $saveOptions['save_and_edit'] = $this->getSaveActionButtonName('save_and_edit');
                break;
        }

        return [
            'active' => $saveCurrent,
            'options' => $saveOptions,
        ];
    }

    /**
     * Get the translated text for the Save button.
     * @param  string $actionValue [description]
     * @return [type]              [description]
     */
    private function getSaveActionButtonName($actionValue = 'save_and_back')
    {
        switch ($actionValue) {
            case 'save_and_edit':
                return trans('backpack::crud.save_action_save_and_edit');
                break;
            case 'save_and_back':
            default:
                return trans('backpack::crud.save_action_save_and_back');
                break;
        }
    }

    /**
     * @param \Backpack\Settings\app\Http\Requests\SettingRequest $request
     *
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Illuminate\Validation\ValidationException
     */
    public function update(UpdateRequest $request)
    {
        $setting = Setting::find($request->input('id'));

        $validation_rules = [];

        switch ($setting->key) {
            case 'sip_server_address':
                $validation_rules['value'] = ['required', new DomainNameOrIp];
                break;
            case 'sip_server_port':
                $validation_rules['value'] = 'required|integer|min:0|max:65535';
                break;
            case 'concierge_sip_number':
                $validation_rules['value'] = 'required';
                break;
        }

        Validator::make($request->all(), $validation_rules)->setAttributeNames([
            'value' => $setting->name,
        ])->validate();

        return parent::update($request);
    }

    protected function setButtons()
    {
        $this->crud->removeAllButtons();

        if (Auth::user()->can(Permission::SETTINGS_EDIT_ACCESS)) {
            $this->crud->addButton('line', 'update', 'view', 'crud::buttons.update');
        }
    }



    /**
     * The search function that is called by the data table.
     *
     * @return  JSON Array of cells in HTML form.
     */
    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }

        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // overwrite any order set in the setup() method with the datatables order
        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);
            if ($column['tableColumn']) {
                // clear any past orderBy rules
                $this->crud->query->getQuery()->orders = null;
                // apply the current orderBy rules
                if(in_array($column['name'], ['name', 'description'])) {
                    // для поля с именем number установлена отдельная сортировка для более удобочитаемого отображения квартир, имеющих номера
                    $this->crud->query = $this->crud->query->orderByRaw(
                        $column['name'] . '+0 '.$column_direction.', '.
                        $column['name'].' '.$column_direction.
                        ', LENGTH('.$column['name'].') '.$column_direction
                    );
                } else {
                    $this->crud->orderBy($column['name'], $column_direction);
                }
            }
        }
        $entries = $this->crud->getEntries();

        foreach ($entries as &$entry) {
            $entry->name = __($entry->name);
            $entry->description = __($entry->description);
        } unset($entry);

        return $this->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);
    }

    /**
     * Created the array to be fed to the data table.
     *
     * @param $entries Eloquent results.
     * @return array
     */
    public function getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex = false)
    {
        $rows = [];

        foreach ($entries as $row) {
            $rows[] = $this->getRowViews($row, $startIndex === false ? false : ++$startIndex);
        }

        return [
            'draw'            => (isset($this->request['draw']) ? (int) $this->request['draw'] : 0),
            'recordsTotal'    => $totalRows,
            'recordsFiltered' => $filteredRows,
            'data'            => $rows,
        ];
    }

    /**
     * Get the HTML of the cells in a table row, for a certain DB entry.
     * @param  Entity $entry A db entry of the current entity;
     * @param  int The number shown to the user as row number (index);
     * @return array         Array of HTML cell contents.
     */
    public function getRowViews($entry, $rowNumber = false)
    {
        $row_items = [];

        foreach ($this->crud->columns as $key => $column) {
            $row_items[] = $this->getCellView($column, $entry, $rowNumber);
        }

        // add the buttons as the last column
        if ($this->crud->buttons->where('stack', 'line')->count()) {
            $row_items[] = \View::make('crud::inc.button_stack', ['stack' => 'line'])
                ->with('crud', $this->crud)
                ->with('entry', $entry)
                ->with('row_number', $rowNumber)
                ->render();
        }

        // add the details_row button to the first column
        if ($this->crud->details_row) {
            $details_row_button = \View::make('crud::columns.details_row_button')
                ->with('crud', $this->crud)
                ->with('entry', $entry)
                ->with('row_number', $rowNumber)
                ->render();
            $row_items[0] = $details_row_button.$row_items[0];
        }

        return $row_items;
    }

    /**
     * Get the HTML of a cell, using the column types.
     * @param  array $column
     * @param  Entity $entry A db entry of the current entity;
     * @param  int The number shown to the user as row number (index);
     * @return HTML
     */
    public function getCellView($column, $entry, $rowNumber = false)
    {
        return $this->renderCellView($this->getCellViewName($column), $column, $entry, $rowNumber);
    }

    /**
     * Render the given view.
     * @param $view
     * @param $column
     * @param $entry
     * @param  int The number shown to the user as row number (index);
     * @return mixed
     */
    private function renderCellView($view, $column, $entry, $rowNumber = false)
    {
        if (! view()->exists($view)) {
            $type = 'text';

            if ($entry->key === 'acs_users' && $column['name'] === 'value') {
                $type = 'boolean';
            }
            $view = 'crud::columns.'.$type; // fallback to text column
        }

        return \View::make($view)
            ->with('crud', $this)
            ->with('column', $column)
            ->with('entry', $entry)
            ->with('rowNumber', $rowNumber)
            ->render();
    }

    /**
     * Get the name of the view to load for the cell.
     * @param $column
     * @return string
     */
    private function getCellViewName($column)
    {
        //dump($column);
        // return custom column if view_namespace attribute is set
        if (isset($column['view_namespace']) && isset($column['type'])) {
            return $column['view_namespace'].'.'.$column['type'];
        }

        if (isset($column['type'])) {
            // if the column has been overwritten return that one
            if (view()->exists('vendor.backpack.crud.columns.'.$column['type'])) {
                return 'vendor.backpack.crud.columns.'.$column['type'];
            }

            if ($column['name'])
                // return the column from the package
                return 'crud::columns.checkbox';
        }

        // fallback to text column
        return 'crud::columns.text';
    }

    public function show($id){
        Abort(404);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     *
     * @return Response
     */
    public function edit($id)
    {
        $this->crud->hasAccessOrFail('update');

        $this->data['entry'] = $entry = $this->crud->getEntry($id);

        $field = json_decode($entry->field, true);

        $field['label'] = __($field['label']);

        $this->crud->addField($field); // <---- this is where it's different
        $this->crud->addField([
            'name'       => 'name',
            'label'      => trans('backpack::settings.name'),
            'type'       => 'text',
            'attributes' => [
                'disabled' => 'disabled',
            ],
            'value' => __($entry->name)
        ]);
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['title'] = trans('backpack::crud.edit').' '.$this->crud->entity_name;

        $this->data['id'] = $id;

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getEditView(), $this->data);
    }
}
