<?php

namespace App\Http\Controllers\Admin;

use App\Models\PermissionManager\Permission;
use App\Traits\Admin\Ordering;
use App\Traits\PermissionManager\CheckingPermissions;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

use Backpack\CRUD\app\Http\Controllers\CrudController;
// VALIDATION
use Backpack\CRUD\app\Http\Requests\CrudRequest as StoreRequest;
use Backpack\CRUD\app\Http\Requests\CrudRequest as UpdateRequest;
use Illuminate\Support\Facades\Validator;

class RoleCrudController extends CrudController
{
    use CheckingPermissions, Ordering;

    /**
     * RoleCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::ROLES_CREATE_ACCESS],
            ['create', 'store'],
        ], [
            [Permission::ROLES_VIEW_ACCESS],
            ['index'],
        ], [
            [Permission::ROLES_EDIT_ACCESS],
            ['edit', 'update'],
        ], [
            [Permission::ROLES_DELETE_ACCESS],
            ['destroy']
        ]]);
    }

    public function setup()
    {
        $this->crud->setModel(config('laravel-permission.models.role'));
        $this->crud->setEntityNameStrings(trans('backpack::permissionmanager.role'), trans('backpack::permissionmanager.roles'));
        $this->crud->setRoute(config('backpack.base.route_prefix').'/role');

        $this->crud->setColumns([
            [
                'name'  => 'name',
                'label' => trans('backpack::permissionmanager.name'),
                'type'  => 'text',
            ],
            [
                // n-n relationship (with pivot table)
                'label'     => ucfirst(trans('backpack::permissionmanager.permission_plural')),
                'type'      => 'select_multiple',
                'name'      => 'permissions', // the method that defines the relationship in your Model
                'entity'    => 'permissions', // the method that defines the relationship in your Model
                'attribute' => 'name', // foreign key attribute that is shown to user
                'model'     => "Backpack\PermissionManager\app\Models\Permission", // foreign key model
                'pivot'     => true, // on create&update, do you need to add/delete pivot table entries?
            ],
        ]);

        $this->crud->addField([
            'name'  => 'name',
            'label' => trans('backpack::permissionmanager.name'),
            'type'  => 'text',
        ]);
        $this->crud->addField([
            'label'     => ucfirst(trans('backpack::permissionmanager.permission_plural')),
            'type'      => 'checklist',
            'name'      => 'permissions',
            'entity'    => 'permissions',
            'attribute' => 'name',
            'model'     => "Backpack\PermissionManager\app\Models\Permission",
            'pivot'     => true,
        ]);

        if (config('backpack.permissionmanager.allow_role_create') == false) {
            $this->crud->denyAccess('create');
        }
        if (config('backpack.permissionmanager.allow_role_update') == false) {
            $this->crud->denyAccess('update');
        }
        if (config('backpack.permissionmanager.allow_role_delete') == false) {
            $this->crud->denyAccess('delete');
        }


        // Make "Permissions" column translatable
        $this->crud->columns['permissions']['attribute'] = 'translated_name';

        // Make "Permissions" checklist field translatable
        $this->crud->create_fields['permissions']['attribute']
            = $this->crud->update_fields['permissions']['attribute']
            = 'translated_name';

        $this->crud->create_fields['permissions']['model']
            = $this->crud->update_fields['permissions']['model']
            = Permission::class;

        $this->applyBackpackPermissions();

        $this->crud->setEditView('crud::roles.edit');
        $this->crud->setCreateView('crud::roles.create');
        $this->setButtons();
    }

    public function update(UpdateRequest $request)
    {
        $this->checkRequest($request, 'update');

        // Подготовить текущий запрос
        $request = $this->setRequest($request);

        return parent::updateCrud($request);
    }

    public function store(StoreRequest $request)
    {
        $this->checkRequest($request);

        // Подготовить текущий запрос
        $request = $this->setRequest($request);

        return parent::storeCrud($request);
    }

    /**
     * Валидация входящего запроса
     *
     */
    protected function checkRequest($request, $action = 'create')
    {
        if ($action === 'create') {
            $name = 'required|unique:roles,name';
        } elseif ($action === 'update') {
            $name = 'required';
        }


        Validator::validate($request->all(), [
            'name'     => $name,
            'permissions' => 'required'
        ], [
            'name.required' => trans('roles.errors.name'),
            'name.unique' => trans('roles.errors.name_unique'),
            'permissions.required' => trans('roles.errors.permissions')
        ]);
    }

    /**
     * Выполнить коррекцию входящего запроса в соответствии с группами прав и установленными ролями.
     *
     */
    protected function setRequest($request)
    {
        $request = $this->setGroupPermissions($request);

        // Преобразовать массив прав в коллекцию и id ролей в тип int
        $request->request->add(['permissions' => collect($request->input('permissions'))->transform(function ($permission) {
            return (int)$permission;
        })->values()->toArray()]);

        return $request;
    }

    /**
     * Устанавливает права из одной группы, обладающие меньшим весом
     *
     */
    private function setGroupPermissions($request)
    {
        // Получить все веса входящих прав из базы данных
        $input_permissions = collect($request->input('permissions'));

        // Проверить правила на наличие полного доступа
        if ($full_access_request = $this->checkFullAccessPermission($request)) {
            return $full_access_request;
        }

        $input_permissions_info = DB::table('permissions')
            ->select('id', 'weight', 'group_id')
            ->whereIn('id', $input_permissions)
            ->get();

        // Найти все права, ниже текущих по весам
        $raw_permissions = collect();
        $input_permissions_info->each(function ($permission) use (&$raw_permissions) {
            $raw_permissions = $raw_permissions->merge(DB::table('permissions')
                ->select('id')
                ->where('weight', '<', $permission->weight)
                ->where('group_id', $permission->group_id)
                ->get()->pluck('id'));
        });

        // Сформировать массив прав, удалив все повторяющиеся
        $permissions = $raw_permissions->merge($input_permissions)->unique();

        // Внести в запрос обновленные данные
        $request->request->add(['permissions' => $permissions]);

        return $request;
    }

    /**
     * Проверка правил на наличие полного доступа
     *
     * @param CrudRequest $request
     * @return CrudRequest|null
     */
    protected function checkFullAccessPermission($request)
    {
        $input_permissions = collect($request->input('permissions'));

        // Проверить на наличие права "полный доступ" в случае его наличия удалить остальные права
        $full_access = Permission::where('name', Permission::FULL_ACCESS)->first();

        // Если есть право "полный доступ", вернуть только его
        if ($input_permissions->contains($full_access->id)) {
            $request->request->add(['permissions' => collect($full_access->id)]);
            return $request;
        }

        return null;
    }

    public function edit($id)
    {
        $this->data['permissions'] = $this->getPermissions()->toJson();

        return parent::edit($id);
    }

    public function create()
    {
        $this->data['permissions'] = $this->getPermissions()->toJson();

        return parent::create();
    }


    /**
     * Получить список прав в виде коллекции
     *
     */
    private function getPermissions()
    {
        $raw_permissions = DB::table('permissions')
            ->select('id', 'group_id AS group', 'weight', 'name')
            ->get();

        $permissions = collect();
        $raw_permissions->each(function ($permission) use (&$permissions) {
            $permission_data = collect(['group' => $permission->group, 'weight' => $permission->weight]);

            if ($permission->name === Permission::FULL_ACCESS) {
                $permission_data->put('full_access', true);
            }

            $permissions->put($permission->id, $permission_data);
        });

        return $permissions;
    }



    /**
     * Get the save configured save action or the one stored in a session variable.
     * @return [type] [description]
     */
    public function getSaveAction()
    {
        $saveAction = session('save_action', config('backpack.crud.default_save_action', 'save_and_back'));
        $saveOptions = [];
        $saveCurrent = [
            'value' => $saveAction,
            'label' => $this->getSaveActionButtonName($saveAction),
        ];

        switch ($saveAction) {
            case 'save_and_edit':
                $saveOptions['save_and_back'] = $this->getSaveActionButtonName('save_and_back');
                if (Auth::user()->can(Permission::ROLES_CREATE_ACCESS)) {
                    $saveOptions['save_and_new'] = $this->getSaveActionButtonName('save_and_new');
                }
                break;
            case 'save_and_new':
                if (Auth::user()->can(Permission::ROLES_CREATE_ACCESS)) {
                    $saveOptions['save_and_back'] = $this->getSaveActionButtonName('save_and_back');
                }
                if (Auth::user()->can(Permission::ROLES_EDIT_ACCESS)) {
                    $saveOptions['save_and_edit'] = $this->getSaveActionButtonName('save_and_edit');
                }
                break;
            case 'save_and_back':
            default:
                if (Auth::user()->can(Permission::ROLES_EDIT_ACCESS)) {
                    $saveOptions['save_and_edit'] = $this->getSaveActionButtonName('save_and_edit');
                }
                if (Auth::user()->can(Permission::ROLES_CREATE_ACCESS)) {
                    $saveOptions['save_and_new'] = $this->getSaveActionButtonName('save_and_new');
                }
                break;
        }

        return [
            'active' => $saveCurrent,
            'options' => $saveOptions,
        ];
    }

    /**
     * Get the translated text for the Save button.
     * @param  string $actionValue [description]
     * @return [type]              [description]
     */
    private function getSaveActionButtonName($actionValue = 'save_and_back')
    {
        switch ($actionValue) {
            case 'save_and_edit':
                return trans('backpack::crud.save_action_save_and_edit');
                break;
            case 'save_and_new':
                return trans('backpack::crud.save_action_save_and_new');
                break;
            case 'save_and_back':
            default:
                return trans('backpack::crud.save_action_save_and_back');
                break;
        }
    }

    protected function setButtons()
    {
        $this->crud->removeAllButtons();

        if (Auth::user()->can(Permission::ROLES_CREATE_ACCESS)) {
            $this->crud->addButton('top', 'create', 'view', 'crud::buttons.create');
        }
        if (Auth::user()->can(Permission::ROLES_DELETE_ACCESS)) {
            $this->crud->addButton('line', 'delete', 'view', 'crud::buttons.delete');
        }
        if (Auth::user()->can(Permission::ROLES_EDIT_ACCESS)) {
            $this->crud->addButton('line', 'update', 'view', 'crud::buttons.update');
        }
    }

    public function show($id){
        Abort(404);
    }

    /**
     * Переопределение необходимо для сортировки ненастраиваемых полей
     *
     * @return array|\Backpack\CRUD\app\Http\Controllers\Operations\JSON
     * @throws \Backpack\CRUD\Exception\AccessDeniedException
     */
    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }

        // structure the response in a DataTable-friendly way
        if (Auth::user()->cannot(Permission::USER_VIEW_FULL_ACCESS)) {
            $subordinateUsersByPermissions = Auth::user()->getSubordinateByPermissions()->pluck('id')->push(Auth::user()->id);
            $this->crud->addClause('whereIn', 'id', $subordinateUsersByPermissions->toArray());
        }

        // Кастомная сортировка
        $sortedEntriesIds = null;
        $sortedEntries = collect();
        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);

            $customOrder = true;
            $table = $this->crud->getModel()->getTable();
            switch ($column['name']) {
                // По столбцу текст
                case 'name':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'name', [
                        'object' => false
                    ]);
                    break;
                default:
                    $customOrder = false;
                    if ($column['tableColumn']) {
                        // clear any past orderBy rules
                        $this->crud->query->getQuery()->orders = null;
                        // apply the current orderBy rules
                        $this->crud->orderBy($column['name'], $column_direction);
                    }
                    break;
            }

            // Если сортировка кастомная, то сформировать коллекцию объектов
            if ($customOrder) {
                $sortedEntries = $this->createOrderedObjectsByIds(config('laravel-permission.models.role'), $sortedEntriesIds);
            }
        }

        // Если сортировка кастомная, то присваиваем отсортированные записи
        $entries = $sortedEntries->count()? $sortedEntries : $this->crud->getEntries();

        return $this->crud->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);
    }
}
