<?php

namespace App\Http\Controllers\Admin;

use App\Models\Apartment;
use App\Models\PermissionManager\Permission;
use App\Models\Review;
use App\Models\Answer;
use App\Models\User;
use App\Notifications\ReviewAnswer;
use App\Notifications\ReviewAnswerDefault;
use App\Traits\Admin\Ordering;
use App\Traits\PermissionManager\CheckingPermissions;
use App\Traits\SetSimpleCookie;
use Backpack\CRUD\app\Http\Controllers\CrudController;

// VALIDATION: change the requests to match your own file names if you need form validation
use App\Http\Requests\ReviewRequest as StoreRequest;
use App\Http\Requests\ReviewRequest as UpdateRequest;
use Illuminate\Support\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\View;

class ReviewCrudController extends CrudController {

    use SetSimpleCookie;
    use CheckingPermissions;
    use Ordering {
        createBuilder as createBuilderParent;
    }

    public static $displayFields = [
        'apartment' => 'text',
        'text' => 'text',
        'answer' => 'text',
        'created_at' => 'text_datetime_utc',
    ];

    /**
     * ReviewCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::REVIEWS_FULL_ACCESS],
            [
                'create',
                'store',
                'index',
                'edit',
                'update',
                'destroy',
                'groupView',
            ],
        ]]);
    }

    public function setup() {

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\Review');
        $this->crud->setRoute(config('backpack.base.route_prefix') . '/review');
        $this->crud->setEntityNameStrings(__('reviews.reviews_singular'), __('reviews.reviews_plural'));

        $this->applyBackpackPermissions();

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */

        // ------ CRUD FIELDS
        $this->crud->addField([
            'name' => 'text',
            'type' => 'text',
            'attributes' => [
                'id' => 'field-text'
            ]
        ], 'update');

        $this->crud->addField([
            'name' => 'answer',
            'type' => 'text',
            'attributes' => [
                'id' => 'field-answer'
            ]
        ], 'update');

        // ------ CRUD COLUMNS

        $this->crud->addClause('where', 'object_id', '=', config('app.object.id'));

        foreach (self::$displayFields as $field => $type) {
            if ($field == 'answer') {
                continue;
            }
            $localizedFieldName = __("reviews.crud_fields.$field");

            $this->crud->addColumn([
                'name' => $field,
                'label' => $localizedFieldName,
                'type' => $type,
            ]);
        }

        $this->crud->addColumn([
            'name' => 'last_answer',
            'label' => __("reviews.crud_fields.answer"),
            'type' => 'model_function',
            'function_name' => 'lastConciergeAnswer',
            'orderable' => true,

        ]);

        $this->crud->addColumn([
            'name' => 'pseudo_status',
            'label' => __('reviews.crud_fields.status'),
            'type' => 'model_function',
            'function_name' => 'getStatusNameHtml',
            'limit' => 500,
            'orderable' => true
        ]);

        // ------ CRUD BUTTONS
        $this->crud->removeAllButtons();

        $this->crud->addButton('line', 'update', 'view', 'crud::buttons.answer', 'end');
        $this->crud->addButtonFromView('line', 'not_viewed', 'not_viewed');


        // possible positions: 'beginning' and 'end'; defaults to 'beginning' for the 'line' stack, 'end' for the others;
        // $this->crud->addButton($stack, $name, $type, $content, $position); // add a button; possible types are: view, model_function
        // $this->crud->addButtonFromModelFunction($stack, $name, $model_function_name, $position); // add a button whose HTML is returned by a method in the CRUD model
        // $this->crud->addButtonFromView($stack, $name, $view, $position); // add a button whose HTML is in a view placed at resources\views\vendor\backpack\crud\buttons
        // $this->crud->removeButton($name);
        // $this->crud->removeButtonFromStack($name, $stack);
        // $this->crud->removeAllButtons();
        // $this->crud->removeAllButtonsFromStack('line');

        // ------ CRUD ACCESS
        // $this->crud->allowAccess(['list', 'create', 'update', 'reorder', 'delete']);
        // $this->crud->denyAccess(['list', 'create', 'update', 'reorder', 'delete']);

        // ------ CRUD REORDER
        // $this->crud->enableReorder('label_name', MAX_TREE_LEVEL);
        // NOTE: you also need to do allow access to the right users: $this->crud->allowAccess('reorder');

        // ------ CRUD DETAILS ROW
        // $this->crud->enableDetailsRow();
        // NOTE: you also need to do allow access to the right users: $this->crud->allowAccess('details_row');
        // NOTE: you also need to do overwrite the showDetailsRow($id) method in your EntityCrudController to show whatever you'd like in the details row OR overwrite the views/backpack/crud/details_row.blade.php

        // ------ REVISIONS
        // You also need to use \Venturecraft\Revisionable\RevisionableTrait;
        // Please check out: https://laravel-backpack.readme.io/docs/crud#revisions
        // $this->crud->allowAccess('revisions');

        // ------ AJAX TABLE VIEW
        // Please note the drawbacks of this though:
        // - 1-n and n-n columns are not searchable
        // - date and datetime columns won't be sortable anymore
        // $this->crud->enableAjaxTable();

        // ------ DATATABLE EXPORT BUTTONS
        // Show export to PDF, CSV, XLS and Print buttons on the table view.
        // Does not work well with AJAX datatables.
        // $this->crud->enableExportButtons();

        // ------ ADVANCED QUERIES
        // $this->crud->addClause('active');
        // $this->crud->addClause('type', 'car');
        // $this->crud->addClause('where', 'name', '==', 'car');
        // $this->crud->addClause('whereName', 'car');
        // $this->crud->addClause('whereHas', 'posts', function($query) {
        //     $query->activePosts();
        // });
        // $this->crud->addClause('withoutGlobalScopes');
        // $this->crud->addClause('withoutGlobalScope', VisibleScope::class);
        // $this->crud->with(); // eager load relationships
        // $this->crud->orderBy();
        // $this->crud->groupBy();
        // $this->crud->limit();
        $this->crud->orderBy('id', 'DESC');

        $this->crud->setUpdateView('crud::reviews.edit');
        $this->crud->setListView('crud::reviews.list');
    }

    public function notViewed($id)
    {
        $this->crud->hasAccessOrFail('update');

        $review = $this->crud->getEntry($id);

        $review->setNotViewed();

        return redirect()->back();
    }

    public function store(StoreRequest $request) {
        // your additional operations before save here
        $redirect_location = parent::storeCrud($request);
        $review = $this->data['entry'];
        $answer = $request->input('answer');
        Answer::create([
            'text' => $answer,
            'apartment' => $review->apartment,
            'review_id' => $review->id,
            'user_id' => $review->user_id,
            'is_concierge' => true,
        ]);
        // your additional operations after save here
        // use $this->data['entry'] or $this->crud->entry
        return $redirect_location;
    }

    public function update($id, UpdateRequest $request) {
        // your additional operations before save here
        $this->crud->hasAccessOrFail('update');
        $review = $this->crud->getEntry($id);
        $answer = $request->input('answer');
        if (empty($answer)) {
            return redirect()->back();
        }
        if (!$review->answer) {
            $review->answer = $answer;
            $review->updated_at_answer = new Carbon;
            $review->save();
        }
        Answer::create([
            'text' => $answer,
            'apartment' => $review->apartment,
            'review_id' => $id,
            'user_id' => $review->user_id,
            'is_concierge' => true,
        ]);

        if ($review->user_id) {
            $arUser = [$review->user_id];
            $push_tokens = (new User)->getPushTokens($arUser, true);

            foreach ($push_tokens as $os_id => $os_tokens) {
                if($os_id) {
                    // is iOS
                    Notification::send($os_tokens, new ReviewAnswerDefault($request->input('answer')));
                } else {
                    // is Android
                    Notification::send($os_tokens, new ReviewAnswer($request->input('answer')));
                }
            }
        }

        return redirect()->back();
    }

    public function chat($id)
    {
        $this->data['entry'] = $this->crud->getEntry($id);
        $this->data['crud'] = $this->crud;
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['chat'] = $this->data['entry']->answers()->get();
        return View::make('vendor.backpack.crud.reviews.show_chat', [
            'crud' => $this->crud,
            'fields' => $this->data['fields'],
            'action' => 'edit',
            'entry' => $this->data['entry'],
            'chat' => $this->data['chat'],
        ])->render();
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     *
     * @return Response
     */
    public function edit($id)
    {
        $this->crud->hasAccessOrFail('update');

        // get the info for that entry
        $this->data['entry'] = $this->crud->getEntry($id);

        // mark review as viewed immediately
        $this->data['entry']->is_viewed = TRUE;
        $this->data['entry']->save();

        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['chat'] = $this->data['entry']->answers()->get();
        if (empty($this->data['chat']->toArray())) {
            $this->updateChat($id);
            $this->data['chat'] = $this->data['entry']->answers()->get();
        }
        $this->data['fields'] = $this->crud->getUpdateFields($id);

        $this->data['title'] = trans('backpack::crud.edit').' '.$this->crud->entity_name;

        $this->data['id'] = $id;
        $this->data['redirect'] = $this->getRedirectParameters(false);
        $this->data['redirect_edit'] = $this->getRedirectParameters();
        $this->data['group'] = $this->getParamsForAnswer(true)? '/group' : '';

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getEditView(), $this->data);
    }

    /**
     * Display all rows in the database for this entity.
     *
     * @return Response
     */
    public function index()
    {
        $this->crud->hasAccessOrFail('list');

        if ($this->setCookie('admin/review', 'true')) {
            return redirect(request()->url());
        }

        $this->data['crud'] = $this->crud;
        $this->data['title'] = ucfirst($this->crud->entity_name_plural);

        // get all entries if AJAX is not enabled
        if (! $this->data['crud']->ajaxTable()) {
            $this->data['entries'] = $this->data['crud']->getEntries();
        }

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getListView(), $this->data);
    }

    private function updateChat($id) {
        $review = Review::find($id);
        if ($review) {
            if (!empty($review->text)) {
                Answer::create([
                    'text' => $review->text,
                    'apartment' => $review->apartment,
                    'review_id' => $id,
                    'user_id' => $review->user_id,
                    'is_concierge' => false,
                ]);
            }
            if (!empty($review->answer)) {
                Answer::create([
                    'text' => $review->answer,
                    'apartment' => $review->apartment,
                    'review_id' => $id,
                    'user_id' => $review->user_id,
                    'is_concierge' => true,
                ]);
            }
        }
    }

    public function groupView(Request $request) {
        $this->crud->hasAccessOrFail('list');

        if ($this->setCookie('admin/review', 'false')) {
            return redirect(request()->url());
        }

        $isFilterApplied = (bool)$request->get('apartment');
        if ($isFilterApplied) {
            $this->crud->addClause('where', 'apartment', '=', $request->get('apartment'));
            
            // Отключить сортировку по квартире при выбранном фильтре
            $this->crud->columns['apartment']['orderable'] = false;
        }

        $this->data['crud'] = $this->crud;

        // build menu tree
        $this->data['tree'] = (new Apartment)->getHouseTree();

        $this->data['badgeNotViewed'] = [];

        $fields = [
            ['name' => 'number', 'group' => '`number`'],
            ['name' =>'entrance', 'group' => '`entrance`'],
            ['name' =>'floor', 'group' => 'CONCAT(`floor`, "-", `entrance`)'],
        ];
        $fieldsStr = implode('`, `', array_column($fields, 'name'));

        foreach ($fields as $field) {
            $groupBy = $field['group'];
            $name = $field['name'];

            $query =
                "
                SELECT 
                  count(*) as `count`, `$fieldsStr`
                FROM (
                  SELECT 
                    `reviews`.`is_viewed`, 
                    `apartments`.`floor`, 
                    `apartments`.`number`, 
                    `apartments`.`entrance` 
                  FROM `reviews` 
                  LEFT JOIN `apartments` 
                  ON `reviews`.`apartment` = `apartments`.`number` and `apartments`.`object_id` = '" . config('app.object.id') . "'
                                                                   and `reviews`.`object_id`    = '" . config('app.object.id') . "'
                  WHERE `reviews`.`is_viewed` = 0 OR `reviews`.`answer` IS NULL
                ) `reviews_not_viewed` 
                GROUP BY $groupBy
                ";

            $dbResult = DB::select($query);
            $result = [];

            if ($name === 'floor') {
                foreach ($dbResult as $item) {
                    $result[$item->entrance][$item->floor] = $item->count;
                }
            } else {
                foreach ($dbResult as $item) {
                    $result[$item->{$name}] = $item->count;
                }
            }

            $this->data['badgeNotViewed'][$name] = $result;

        }

        $this->data['selected'] = $request->all();

        // get all entries if AJAX is not enabled
        // always show all entries on group view if filter is applied
        if ($isFilterApplied || !$this->data['crud']->ajaxTable()) {
            $this->data['entries'] = $this->data['crud']->getEntries();
        }

        return view('crud::reviews.group', $this->data);
    }

    public function show($id){
       Abort(404);
    }



/*
 *  ---------------------------------------Search start---------------------------------------
 *
 *  Переопределения методов стандартного поиска для решения определенных задач
 */
    public function search()
    {
        // Замыкание для отображения выбранной квартиры на страницы группировки
        $apartment = $this->request->get('apartment');

        if (is_string($apartment) && Review::whereRaw('object_id = ?, apartment = ?', [config('app.object.id'), $apartment])) {
            $this->crud->addClause('where', 'apartment', '=', $apartment);
            $this->crud->addClause('where', 'object_id', '=', config('app.object.id'));
        }

        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // overwrite any order set in the setup() method with the datatables order
        $rawSortedEntries = collect();
        if ($this->request->input('order')) {
            $customOrder = false;

            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);
            if ($column['tableColumn']) {
                // clear any past orderBy rules
                $this->crud->query->getQuery()->orders = null;
                // apply the current orderBy rules
                if(in_array($column['name'], ['apartment', 'text', 'answer'])) {
                    // для поля с именем number установлена отдельная сортировка для более удобочитаемого отображения квартир, имеющих номера
                    $this->crud->query = $this->crud->query->orderByRaw(
                        $column['name'] . '+0 '.$column_direction.', '.
                        $column['name'].' '.$column_direction.
                        ', LENGTH('.$column['name'].') '.$column_direction
                    );
                } else {
                    $this->crud->orderBy($column['name'], $column_direction);
                }
            } elseif ($column['name'] === 'pseudo_status') {
                $table = $this->crud->getModel()->getTable();
                $rawSortedEntries = $this->sortBySimpleRow($table, 'status');
                $customOrder = true;
            }

            // Если сортировка кастомная, то сформировать коллекцию объектов
            if ($customOrder) {
                $rawSortedEntries = $this->createOrderedObjectsByIds('App\Models\Review', $rawSortedEntries);
            }
        }
        // Если сортировка кастомная, то присваиваем отсортированные записи
        $entries = $rawSortedEntries->count()? $rawSortedEntries : $this->crud->getEntries();

        $data = $this->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);

        /* Поиск и выделение жирным активных записей
         *
         * Сформировать массив идентификаторов, которые будут выделяться жирным
         * answered - которые будут выделяться жирным
         */

        $answered = collect();

        for ($i = 0; $i < count($entries); $i++) {
            if (empty($entries[$i]->answer) || !$entries[$i]->is_viewed ||
                empty($entries[$i]->lastAnswer()->first()->is_concierge)
            ) {
                $answered->push($i);
            }
        }

        foreach ($answered as $answer) {
            $data['data'][$answer]['DT_RowClass'] = 'font-weight-900';
        }

        return $data;
    }

    /**
     * Перегрузка необходима для добавления параметров к кнопке "Ответить"
     *
     * Created the array to be fed to the data table.
     *
     * @param $entries Eloquent results.
     * @return array
     */
    public function getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex = false)
    {
        $rows = [];

        // Составить строку из параметров, если они присутствуют
        $get_result = $this->getRedirectParameters();

        foreach ($entries as $row) {
            $rows[] = $this->getRowViews($row, $startIndex === false ? false : ++$startIndex, $get_result);
        }

        return [
            'draw'            => (isset($this->request['draw']) ? (int) $this->request['draw'] : 0),
            'recordsTotal'    => $totalRows,
            'recordsFiltered' => $filteredRows,
            'data'            => $rows,
        ];
    }

    /**
     * Перегрузка необходима для добавления параметров к кнопке "Ответить"
     *
     * Get the HTML of the cells in a table row, for a certain DB entry.
     * @param  Entity $entry A db entry of the current entity;
     * @param  int The number shown to the user as row number (index);
     * @return array         Array of HTML cell contents.
     */
    public function getRowViews($entry, $rowNumber = false, $get_params = null)
    {
        $row_items = [];

        foreach ($this->crud->columns as $key => $column) {
            $row_items[] = $this->crud->getCellView($column, $entry, $rowNumber);
        }

        // add the buttons as the last column
        if ($this->crud->buttons->where('stack', 'line')->count()) {
            $row_items[] = \View::make('crud::inc.button_stack', ['stack' => 'line'])
                ->with('crud', $this->crud)
                ->with('entry', $entry)
                ->with('parameters', $get_params)
                ->with('row_number', $rowNumber)
                ->render();
        }

        return $row_items;
    }
/*
 *  ---------------------------------------Search end---------------------------------------
 */

    /**
     * Получить параметры для корректной переадресации (например, вернуться на последнюю открытую страницу
     *
     * @return null|string
     */
    protected function getRedirectParameters(bool $group = true)
    {
        // Определить параметры для переадресации
        $params = $this->getParamsForAnswer();

        // Составить строку из параметров, если они присутствуют
        $get_params = [];
        if ($params['group'] && $group) {
            $get_params[] = 'group=true';
        }
        if ($params['number']) {
            $get_params[] = 'apartment='.$params['number'];
        }
        if ($params['floor']) {
            $get_params[] = 'floor='.$params['floor'];
        }
        if ($params['entrance']) {
            $get_params[] = 'entrance='.$params['entrance'];
        }

        $get_result = null;
        if (count($get_params)) {
            $get_result = '?'.implode('&', $get_params);
        }

        return $get_result;
    }

    /**
     * Получение списка параметров для переадресации
     *
     * @return array
     */
    protected function getParamsForAnswer(bool $onlyGroup = false)
    {
        // Проверка параметра группы
        $notGroup = $this->getCookie('admin/review');

        if ($notGroup === 'true') {
            $group = false;
        } else {
            $group = true;
        }
        if ($onlyGroup) {
            return $group;
        }

        // Проверка параметра квартиры
        return [
            'group'     => $group,
            'number'    => $this->request->get('apartment'),
            'floor'     => $this->request->get('floor'),
            'entrance'  => $this->request->get('entrance')
        ];
    }

    /**
     * Переопределение построителя для подключения таблицы категорий
     *
     * @param $table
     * @param null $selectRaw
     * @param bool $object
     * @param null $clause
     * @return \Illuminate\Database\Query\Builder
     */
    protected function createBuilder($table, $selectRaw = null, $object = true, $clause = null)
    {
        $raw_builder = $this->createBuilderParent($table, $selectRaw, $object, $clause);

        // Так как стату отсутствует в таблице, было добавлено фейковое поле статус, которое строиться по тому же самому принипу
        $raw_builder->selectRaw('
        IF(
            answer IS NOT NULL, 2,
            IF(
                is_viewed <> 0, 1, 0
            )
        ) AS status');

        $isFilterApplied = (bool)$this->request->get('apartment');
        if ($isFilterApplied) {
            $raw_builder->where('apartment', $this->request->get('apartment'));
        }

        return $raw_builder;
    }
}
