<?php

namespace App\Http\Controllers\Admin;

use App\Models\PermissionManager\Permission;
use App\Models\QuizAnswer;
use App\Models\QuizTemplate;
use App\Traits\Admin\Ordering;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\CRUD\app\Http\Controllers\CrudController;
use App\Http\Requests\QuizTemplateRequest as StoreRequest;
use App\Http\Requests\QuizTemplateRequest as UpdateRequest;
use App\Models\Category;
use Backpack\CRUD\app\Http\Requests\CrudRequest;
use Illuminate\Support\Facades\Validator;

class QuizTemplateCrudController extends CrudController
{
    use CheckingPermissions, Ordering {
        createBuilder as createBuilderParent;
    }

    /**
     * QuizTemplateCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::QUIZ_TEMPLATES_ACCESS],
            [
                'create',
                'store',
                'index',
                'edit',
                'update',
                'destroy',
            ],
        ]]);
    }

    public function setup()
    {

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\QuizTemplate');
        $this->crud->setRoute(config('backpack.base.route_prefix') . '/templates/quizzes');
        $this->crud->setEntityNameStrings(__('quizzes.quiz_template_singular'), __('quizzes.quiz_template_plural'));

        $this->applyBackpackPermissions();

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */

        $this->crud->addColumn([
            'name' => 'title',
            'label' => __('quizzes.crud_fields.title'),
            'type' => 'text',
        ]);
        $this->crud->addColumn([
            'label' => __('templates.messages.crud_fields.category_id'),
            'type' => 'select',
            'name' => 'category_id',
            'entity' => 'category',
            'attribute' => 'name',
            'model' => "App\Models\Category",
        ]);


        // fields for forms
        $this->crud->addField([
            'name' => 'title',
            'label' => __('quizzes.crud_fields.title'),
            'type' => 'text',
        ]);

        $this->crud->addField([
            'name' => 'question',
            'label' => __('quizzes.crud_fields.question'),
            'type' => 'textarea',
        ]);

        $this->crud->addField([
            'name' => 'answers',
            'label' => __('quizzes.crud_fields.variants'),
            'type' => 'table',
            'columns' => ['text' => ''],
            'entity_singular' => __('quizzes.crud_fields.variant_add_button'), // used on the "Add X" button
            'max' => 20, // maximum rows allowed in the table
            'min' => 0 // minimum rows allowed in the table
        ]);

        $this->crud->addField([
            'name' => 'category',
            'value' => '<i class="fa fa-folder-open" aria-hidden="true"></i>&nbsp;&nbsp;
                <span class="chosen-category">' . __('templates.categories.category_empty') . '</span>&nbsp;&nbsp;
                <button class="btn btn-primary" type="button" data-toggle="modal" data-target="#categories-picker">'
                . __('templates.categories.buttons.view') . '</button>',
            'type' => 'custom_html',
        ]);

        $this->crud->addField([
            'name' => 'category_id',
            'type' => 'hidden',
            'attributes' => [
                'id' => 'field-category_id'
            ],
        ]);

        $this->crud->addField([
            'name' => 'free_answer',
            'label' => __('quizzes.crud_fields.free_answer'),
            'type' => 'checkbox',
        ]);

        $this->crud->addField([
            'name' => 'multiple',
            'label' => __('quizzes.crud_fields.multiple'),
            'type' => 'checkbox',
        ]);

        $templates = QuizTemplate::where('tags','!=', null)->get()->pluck('tags')->toArray();
        $tags = [];
        foreach ($templates as $template){
            $tags = array_merge(explode(',',$template), $tags);
        }

        $tags = array_values(array_unique($tags));

        $this->crud->addField([
            'name' => 'tags',
            'label' => __('templates.quiz.crud_fields.tags'),
            'type' => 'tags',
            'variants' => \GuzzleHttp\json_encode($tags)
        ]);


        $this->crud->addField([
            'name' => 'is_template',
            'label' => '',
            'type' => 'hidden',
            'value' => 1
        ]);

        $this->crud->setCreateView('crud::templates.quiz.create');
        $this->crud->setEditView('crud::templates.quiz.edit');

    }

    public function store(CrudRequest $request)
    {
        $this->crud->hasAccessOrFail('create');

        $request->request->add(['object_id' => null]);

        if ($reinput = $this->validation($request)) {
            return $reinput;
        }

        // fallback to global request instance
        if (is_null($request)) {
            $request = \Request::instance();
        }

        // replace empty values with NULL, so that it will work with MySQL strict mode on
        foreach ($request->input() as $key => $value) {
            if (empty($value) && $value !== '0') {
                $request->request->set($key, null);
            }
        }

        // insert item in the db
        $item = $this->crud->create($request->except(['save_action', '_token', '_method']));
        $this->data['entry'] = $this->crud->entry = $item;

        // show a success message
        \Alert::success(trans('backpack::crud.insert_success'))->flash();

        // save the redirect choice for next time
        $this->setSaveAction();

        $this->updateOrCreateVariants(json_decode($request->input('answers')));

        return $this->performSaveAction($item->getKey());
    }

    public function update(CrudRequest $request)
    {
        $this->crud->hasAccessOrFail('update');

        $request->request->add(['object_id' => null]);

        $issetModelsAnswers = (new QuizAnswer())->where('quiz_id', $request->input('id'))->get()->toArray();

        if ($reinput = $this->validation($request)) {
            return $reinput;
        }

        // fallback to global request instance
        if (is_null($request)) {
            $request = \Request::instance();
        }

        // replace empty values with NULL, so that it will work with MySQL strict mode on
        foreach ($request->input() as $key => $value) {
            if (empty($value) && $value !== '0') {
                $request->request->set($key, null);
            }
        }

        // update the row in the db
        $item = $this->crud->update($request->get($this->crud->model->getKeyName()), $request->except('save_action', '_token', '_method'));
        $this->data['entry'] = $this->crud->entry = $item;

        // show a success message
        \Alert::success(trans('backpack::crud.update_success'))->flash();

        // save the redirect choice for next time
        $this->setSaveAction();

        $this->updateOrCreateVariants(json_decode($request->input('answers')),$issetModelsAnswers);

        return $this->performSaveAction($item->getKey());
    }

    /**
     * Show the form for creating inserting a new row.
     *
     * @return Response
     */
    public function create() {
        $this->crud->hasAccessOrFail('create');
        // prepare the fields you need to show
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getCreateFields();
        $this->data['title'] = trans('backpack::crud.add') . ' ' . $this->crud->entity_name;
        $this->data['categories'] = collect(Category::all())
            ->sortBy('lft')
            ->keyBy((new Category)->getKeyName());
        //        dd($this->data);

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getCreateView(), $this->data);
    }

    public function edit($id) {
        $this->crud->hasAccessOrFail('update');

        // prepare the fields you need to show
        // get the info for that entry
        $this->data['entry'] = $this->crud->getEntry($id);
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['title'] = trans('backpack::crud.edit').' '.$this->crud->entity_name;
        $this->data['id'] = $id;
        $this->data['categories'] = collect(Category::all())
            ->sortBy('lft')
            ->keyBy((new Category)->getKeyName());

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getEditView(), $this->data);
    }

    /**
     * Update quiz answers variants or create new
     *
     * @param $issetAnswers
     * @param $newAnswers
     */
    private function updateOrCreateVariants($newAnswers, $issetAnswers = []) {

        // in request answers can be null, but parameter is always sent to method, so we can't use default value
        $newAnswers = $newAnswers ?: [];

        $arIds = [];
        $variant = new QuizAnswer;
        foreach($newAnswers as $answer) {
            if($answer->text){
                if(!isset($answer->id)) {
                    $variant->create(['quiz_id' => $this->crud->entry->id, 'text' => $answer->text]);
                    $arIds[] = $variant->id;
                } else {
                    $variant = $variant->find($answer->id);
                    $variant->text = $answer->text;
                    $variant->save();
                    $arIds[] = $variant->id;
                }
            }
        }
        if($issetAnswers) {
            foreach ($issetAnswers as $issetAnswer) {
                if (!in_array($issetAnswer['id'], $arIds)) {
                    $variant->find($issetAnswer['id'])->delete();
                }
            }
        }
    }

    /*
     * Валидация для страницы опросов
     */
    protected function validation(CrudRequest $request)
    {
        // Валидатор для основного контекста
        $main_validator = Validator::make($request->all(), [
            'title' => 'required|max:100',
            'category_id' => 'required',
            'question' => 'required|max:1000',
            'answers' => 'required_if:free_answer,0'
        ]);

        $errors = [];
        $main_validator_fails = $main_validator->fails();

        if ($main_validator_fails) {
            $errors = array_merge($errors, $main_validator->errors()->toArray());
        }

        // Подготовка к валидации таблицы вопросов
        $answers = json_decode($request->input('answers'));
        $variants_json = $answers? $answers : [];

        $variants_rules = [];
        $variants = [];
        foreach ($variants_json as $key => $variant) {
            if (isset($variant->text)) {
                $variants[$key] = $variant->text;
            } else {
                $variants[$key] = '';
            }
            $variants_rules[$key] = 'required|string';
        }

        // Валидация таблицы вопросов
        $variants_validator = Validator::make($variants, $variants_rules);

        $variants_validator_fails = $variants_validator->fails();
        if ($variants_validator_fails) {
            $errors['variants'] = [trans('quizzes.errors.variants')];
        }

        // Если хотябы одна из валидаций не прошла, возвращается обратно с ошибкой
        if ($main_validator_fails || $variants_validator_fails) {
            return redirect()->back()->withErrors($errors)->withInput();
        }

        else return null;
    }

    public function show($id){
        Abort(404);
    }



    /**
     * Переопределение необходимо для сортировки ненастраиваемых полей
     *
     * @return array|\Backpack\CRUD\app\Http\Controllers\Operations\JSON
     * @throws \Backpack\CRUD\Exception\AccessDeniedException
     */
    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // Кастомная сортировка
        $sortedEntriesIds = null;
        $sortedEntries = collect();
        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);

            $customOrder = true;
            $table = $this->crud->getModel()->getTable();
            switch ($column['name']) {
                // По столбцу текст
                case 'title':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'quizzes.title', [
                        'object' => false
                    ]);
                    break;
                // По столбцу тема
                case 'category_id':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'categories.name', [
                        'object' => false
                    ]);
                    break;
                default:
                    $customOrder = false;
                    if ($column['tableColumn']) {
                        // clear any past orderBy rules
                        $this->crud->query->getQuery()->orders = null;
                        // apply the current orderBy rules
                        $this->crud->orderBy($column['name'], $column_direction);
                    }
                    break;
            }

            // Если сортировка кастомная, то сформировать коллекцию объектов
            if ($customOrder) {
                $sortedEntries = $this->createOrderedObjectsByIds('App\Models\QuizTemplate', $sortedEntriesIds);
            }
        }

        // Если сортировка кастомная, то присваиваем отсортированные записи
        $entries = $sortedEntries->count()? $sortedEntries : $this->crud->getEntries();

        return $this->crud->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);
    }


    /**
     * Переопределение построителя для подключения таблицы категорий
     *
     * @param $table
     * @param null $selectRaw
     * @param bool $object
     * @param null $clause
     * @return \Illuminate\Database\Query\Builder
     */
    protected function createBuilder($table, $selectRaw = null, $object = true, $clause = null)
    {
        $raw_builder = $this->createBuilderParent($table, $selectRaw, $object, $clause);

        $raw_builder
            ->join('categories', 'categories.id', '=', $this->crud->getModel()->getTable(). '.category_id')
            ->select($this->crud->getModel()->getTable() . '.*', 'categories.name');

        return $raw_builder;
    }
}
