<?php

namespace App\Http\Controllers\Admin;

use App\Models\Apartment;
use App\Models\Category;
use App\Models\MessageTemplate;
use App\Models\PermissionManager\Permission;
use App\Models\User;
use App\Notifications\NewNotice;
use App\Traits\Admin\Ordering;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\CRUD\app\Http\Controllers\CrudController;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;

// VALIDATION: change the requests to match your own file names if you need form validation
use App\Http\Requests\NoticeRequest as StoreRequest;
use App\Http\Requests\NoticeRequest as UpdateRequest;

class NoticeCrudController extends CrudController {

    use CheckingPermissions, Ordering;

    /**
     * NoticeCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::NOTIFICATIONS_FULL_ACCESS],
            [
                'create',
                'store',
                'index',
                'edit',
                'update',
                'destroy',
            ],
        ]]);
    }

    /**
     * @var string[] $displayFields
     */
    public static $displayFields = [
        'theme',
        'text',
        'created_at',
        'actual_from_date',
        'actual_to_date',
    ];

    public function setup() {

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\Notice');
        $this->crud->setRoute(config('backpack.base.route_prefix') . '/notice');
        $this->crud->setEntityNameStrings(__('notice.notice_singular'), __('notice.notice_plural'));

        $this->applyBackpackPermissions();

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */

        // ------ CRUD FIELDS

        $this->crud->addField([
            'name' => 'object_id',
            'type' => 'hidden',
            'value' => config('app.object.id')
        ]);

        $this->crud->addField([
            'name' => 'destination',
            'label' => __('adverts.form_fields.destination'),
            'type' => 'hidden',
            'attributes' => [
                'id' => 'destination-field',
            ],
        ]);

        $this->crud->addField([
            'name' => 'destinations-list',
            'type' => 'custom_html',
            'is_destination' => TRUE,
            'value' => '<label>' . __('adverts.form_fields.destination') . '</label><div class="destinations-list">
            <h4><label class="label label-primary">' . __('notice.form_fields.apartment') . ' </label></h4></div>',
        ], 'update');

        $apartment = new Apartment;

        $this->crud->addField([
            'name' => 'apartment',
            'label' => __('notice.form_fields.apartment'),
            'type' => 'select2_from_array',
            'options' => $apartment->getFlats(),
            'attributes' => [
                'id' => 'apartment-field',
            ],
        ], 'create');

        $this->crud->addField([
            'name' => 'apartment',
            'label' => __('notice.form_fields.apartment'),
            'type' => 'hidden',
            'attributes' => [
                'id' => 'apartment-field',
            ],
        ], 'update');

        $this->crud->addField([
            'name' => 'advert_date_range',
            'start_name' => 'actual_from_date',
            'end_name' => 'actual_to_date',
            'label' => __('notice.crud_fields.notice_date_range'),
            'type' => 'date_range',
            'start_default' => date('Y-m-d'),
            'end_default' => date('Y-m-d', strtotime('+1 day')),
            'date_range_options' => [
                'locale' => ['format' => 'DD.MM.YYYY'],
            ],
        ], 'create');

        $this->crud->addField([
            'name' => 'actual_from_date',
            'label' => __('notice.crud_fields.actual_from_date'),
            'type' => 'text',
            'attributes' => [
                'id' => 'actual_from_date-field',
                'readonly' => 'true',
            ],
        ], 'update');

        $this->crud->addField([
            'name' => 'actual_to_date',
            'label' => __('notice.crud_fields.actual_to_date'),
            'type' => 'text',
            'attributes' => [
                'id' => 'actual_to_date-field',
                'readonly' => 'true',
            ],
        ], 'update');

        $this->crud->addField([
            'name' => 'template',
            'value' => '<button class="btn btn-primary" type="button" data-toggle="modal" data-target="#templates-picker">'
                . __('adverts.buttons.template') . '</button>',
            'type' => 'custom_html',
        ]);

        $this->crud->addField([
            'name' => 'theme',
            'label' => __('notice.crud_fields.theme'),
            'type' => 'text',
            'attributes' => [
                'id' => 'theme-field',
            ],
        ]);

        $this->crud->addField([
            'name' => 'text',
            'label' => __('notice.crud_fields.text'),
            'type' => 'textarea',
            'attributes' => [
                'id' => 'text-field',
            ],
        ]);

        // ------ CRUD COLUMNS

        $this->crud->addClause('where', 'object_id', '=', config('app.object.id'));

        $this->crud->addColumn([
            'name' => 'apartment',
            'label' => __('notice.crud_fields.whom'),
            'type' => 'model_function',
            'function_name' => 'getWhom'
        ]);

        foreach (self::$displayFields as $field) {
            $localizedFieldName = __("notice.crud_fields.$field");

            $columnType = $field == 'created_at' ? 'text_datetime_utc' : 'text';

            $this->crud->addColumn([
                'name' => $field,
                'label' => $localizedFieldName,
                'type' => $columnType,
            ]);
        }

        // ------ CRUD BUTTONS
        // possible positions: 'beginning' and 'end'; defaults to 'beginning' for the 'line' stack, 'end' for the others;
        // $this->crud->addButton($stack, $name, $type, $content, $position); // add a button; possible types are: view, model_function
        // $this->crud->addButtonFromModelFunction($stack, $name, $model_function_name, $position); // add a button whose HTML is returned by a method in the CRUD model
        // $this->crud->addButtonFromView($stack, $name, $view, $position); // add a button whose HTML is in a view placed at resources\views\vendor\backpack\crud\buttons
        // $this->crud->removeButton($name);
        // $this->crud->removeButtonFromStack($name, $stack);
        // $this->crud->removeAllButtons();
        // $this->crud->removeAllButtonsFromStack('line');

        // ------ CRUD ACCESS
        $this->crud->allowAccess(['list', 'create', 'show']);
        // $this->crud->denyAccess(['list', 'create', 'update', 'reorder', 'delete']);

        // ------ CRUD REORDER
        // $this->crud->enableReorder('label_name', MAX_TREE_LEVEL);
        // NOTE: you also need to do allow access to the right users: $this->crud->allowAccess('reorder');

        // ------ CRUD DETAILS ROW
        // $this->crud->enableDetailsRow();
        // NOTE: you also need to do allow access to the right users: $this->crud->allowAccess('details_row');
        // NOTE: you also need to do overwrite the showDetailsRow($id) method in your EntityCrudController to show whatever you'd like in the details row OR overwrite the views/backpack/crud/details_row.blade.php

        // ------ REVISIONS
        // You also need to use \Venturecraft\Revisionable\RevisionableTrait;
        // Please check out: https://laravel-backpack.readme.io/docs/crud#revisions
        // $this->crud->allowAccess('revisions');

        // ------ AJAX TABLE VIEW
        // Please note the drawbacks of this though:
        // - 1-n and n-n columns are not searchable
        // - date and datetime columns won't be sortable anymore
        // $this->crud->enableAjaxTable();

        // ------ DATATABLE EXPORT BUTTONS
        // Show export to PDF, CSV, XLS and Print buttons on the table view.
        // Does not work well with AJAX datatables.
        // $this->crud->enableExportButtons();

        // ------ ADVANCED QUERIES
        // $this->crud->addClause('active');
        // $this->crud->addClause('type', 'car');
        // $this->crud->addClause('where', 'name', '==', 'car');
        // $this->crud->addClause('whereName', 'car');
        // $this->crud->addClause('whereHas', 'posts', function($query) {
        //     $query->activePosts();
        // });
        // $this->crud->addClause('withoutGlobalScopes');
        // $this->crud->addClause('withoutGlobalScope', VisibleScope::class);
        // $this->crud->with(); // eager load relationships
        // $this->crud->orderBy();
        // $this->crud->groupBy();
        // $this->crud->limit();
        $this->crud->removeButton('delete');
        $this->crud->removeButton('update');
        $this->crud->removeButton('preview');
        $this->crud->addButton('line', 'show', 'view', 'vendor.backpack.crud.notice.buttons.preview', 'top');
        $this->crud->orderBy('created_at', 'DESC');
        // ------ CRUD CUSTOM VIEWS
        $this->crud->setCreateView('crud::notice.create');
        $this->crud->setEditView('crud::notice.edit');
        $this->crud->setShowView('crud::notice.show');
    }

    public function store(StoreRequest $request) {
        parent::storeCrud($request);

        return \Redirect::to($this->crud->route);
    }

    public function update(UpdateRequest $request) {
        // your additional operations before save here
        parent::updateCrud($request);
        // your additional operations after save here
        // use $this->data['entry'] or $this->crud->entry
        return \Redirect::to($this->crud->route);
    }

    public function show($id) {
        $this->crud->hasAccessOrFail('show');
        $this->data['entry'] = $this->crud->getEntry($id);
        $this->data['crud'] = $this->crud;
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['title'] = trans('backpack::crud.preview') . ' ' . $this->crud->entity_name;
        unset($this->data['fields']['template']);
        foreach ($this->data['fields'] as $key => &$field) {
            $field['attributes']['readonly'] = true;
        }
        $this->data['id'] = $id;
        return view($this->crud->getShowView(), $this->data);
    }

    /**
     * Show the form for creating inserting a new row.
     *
     * @return Response
     */
    public function create() {
        $this->crud->hasAccessOrFail('create');

        // prepare the fields you need to show
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getCreateFields();
        $this->data['title'] = trans('backpack::crud.add') . ' ' . $this->crud->entity_name;

        $this->setTemplatesData();

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getCreateView(), $this->data);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     *
     * @return Response
     */
    public function edit($id) {
        $this->crud->hasAccessOrFail('update');

        // get the info for that entry
        $this->data['entry'] = $this->crud->getEntry($id);
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['title'] = trans('backpack::crud.edit') . ' ' . $this->crud->entity_name;


        $this->data['id'] = $id;

        $this->setTemplatesData();

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getEditView(), $this->data);
    }

    protected function setTemplatesData() {
        $this->data['templatePickerTitle'] = trans('adverts.add_template');
        $this->data['templateStyle'] = Auth::user()->getStyleType();

        switch ($this->data['templateStyle']) {
            case 'list':
                $this->data['templates'] = MessageTemplate::all();

                break;
            case 'tags':
                $templateTags = MessageTemplate::where('tags', '!=', NULL)
                    ->get()
                    ->pluck('tags')
                    ->toArray();
                $tags = [];
                foreach ($templateTags as $tplTag) {
                    $tags = array_merge(explode(',', $tplTag), $tags);
                }
                $this->data['tags'] = array_values(array_unique($tags));

                $this->data['templates'] = MessageTemplate::all();

                break;

            case 'tree':
                $this->data['categories'] = collect(Category::all())
                    ->sortBy('lft')
                    ->keyBy((new Category)->getKeyName());
                foreach ($this->data['categories'] as $key => $entry) {
                    $entry->items = $entry->messageTemplates()->get();
                }
                break;
        }
    }

    // Перегрузка метода для поиска и выделения жирным активных записей
    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // Кастомная сортировка
        $sortedEntriesIds = null;
        $sortedEntries = collect();
        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);

            $customOrder = true;
            $table = $this->crud->getModel()->getTable();
            switch ($column['name']) {
                // По столбцу тема
                case 'theme':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'theme');
                    break;
                // По столбцу объявление
                case 'text':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'text');
                    break;
                // По столбцу квартира
                case 'apartment':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'apartment');
                    break;
                default:
                    $customOrder = false;
                    if ($column['tableColumn']) {
                        // clear any past orderBy rules
                        $this->crud->query->getQuery()->orders = null;
                        // apply the current orderBy rules
                        $this->crud->orderBy($column['name'], $column_direction);
                    }
                    break;
            }

            // Если сортировка кастомная, то сформировать коллекцию объектов
            if ($customOrder) {
                $sortedEntries = $this->createOrderedObjectsByIds('App\Models\Notice', $sortedEntriesIds);
            }
        }

        // Если сортировка кастомная, то присваиваем отсортированные записи
        $entries = $sortedEntries->count()? $sortedEntries : $this->crud->getEntries();

        $data = $this->crud->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);

        /* Сформировать массив идентификаторов, которые будут выделяться жирным
         * isActive - которые будут выделяться жирным
         * isWillActive - которые будут выделяться очень жирным
         */
        $entries = $this->crud->getEntries();

        $isActive = collect();
        $isWillActive = collect();

        for ($i = 0; $i < count($entries); $i++) {
            if ($entries[$i]->isActive()) {
                $isActive->push($i);
            } elseif ($entries[$i]->isWillActive()) {
                $isWillActive->push($i);
            }
        }

        foreach ($isActive as $activeId) {
            $data['data'][$activeId]['DT_RowClass'] = 'font-weight-900';
        }

        foreach ($isWillActive as $willActiveId) {
            $data['data'][$willActiveId]['DT_RowClass'] = 'font-weight-600';
        }

        return $data;
    }
}
