<?php

namespace App\Http\Controllers\Admin;

use App\Models\ApartmentSipNumber;
use Illuminate\Support\Facades\Auth;
use Storage;
use App\Models\Apartment;
use App\Models\Claim;
use App\Models\PermissionManager\Permission;
use App\Models\User;
use App\Models\ClaimImage;
use App\Notifications\ClaimStatusChanging;
use App\Notifications\ClaimStatusChangingDefault;
use App\Traits\PermissionManager\CheckingPermissions;
use App\Traits\SetSimpleCookie;
use Backpack\CRUD\app\Http\Controllers\CrudController;

// VALIDATION: change the requests to match your own file names if you need form validation
use App\Http\Requests\ClaimRequest as StoreRequest;
use App\Http\Requests\ClaimRequest as UpdateRequest;
use Illuminate\Support\Facades\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;


class ClaimCrudController extends CrudController {

    use SetSimpleCookie;
    use CheckingPermissions;

    /**
     * ClaimCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::CLAIMS_ALL_ACCESS, Permission::CLAIMS_BY_TYPE_ACCESS],
            [
                'create',
                'store',
                'index',
                'edit',
                'update',
                'destroy',
                'groupView',
                'acceptClaim',
                'rejectClaim',
                'closeClaim',
            ],
        ]]);
    }

    /**
     * @throws \Spatie\Permission\Exceptions\PermissionDoesNotExist
     */
    public function setup() {

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\Claim');
        $this->crud->setRoute(config('backpack.base.route_prefix') . '/claims');
        $this->crud->setEntityNameStrings(__('claims.claim'), __('claims.claims'));

        $this->applyBackpackPermissions();

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */

        $this->crud->setDefaultPageLength(20);

        // ------ CRUD COLUMNS

        $this->crud->addClause('where', 'object_id', '=', config('app.object.id'));

        if (!Auth::user()->hasPermissionTo(Permission::CLAIMS_ALL_ACCESS)) {
            $this->crud->addClause('whereIn', 'type_id', auth()->user()->claimTypes->pluck('id'));
        }

        $this->crud->addColumn([
            'name' => 'apartment_number',
            'label' => __('claims.crud_fields.flat'),
            'type' => 'text',
            'orderable' => true
        ]);
        $this->crud->addColumn([
            'name' => 'created_at',
            'label' => __('claims.crud_fields.created_at'),
            'type' => 'text_datetime_utc',
        ]);
        $this->crud->addColumn([
            'name' => 'type',
            'label' => __('claims.crud_fields.type'),
            'type' => 'model_function',
            'function_name' => 'getTypeText',
            'orderable' => true
        ]);
        $this->crud->addColumn([
            'name' => 'claim_text',
            'label' => __('claims.crud_fields.claim_text'),
            'type' => 'model_function_unlimited',
            'function_name' => 'getClaimText',
            'limit' => null
        ]);
        $this->crud->addColumn([
            'name' => 'need_at',
            'label' => __('claims.crud_fields.need_at'),
            'type' => 'text',
        ]);
        $this->crud->addColumn([
            'name' => 'phone',
            'label' => __('claims.crud_fields.phone'),
            'type' => 'text',
        ]);
        $this->crud->addColumn([
            'name' => 'status',
            'label' => __('claims.crud_fields.status'),
            'type' => 'model_function_unlimited',
            'function_name' => 'getStatusNameHtml',
        ]);
        $this->crud->addColumn([
            'name' => 'images',
            'label' => __('claims.crud_fields.images'),
            'type' => 'image_group',
        ]);
        // ------ CRUD BUTTONS
        // possible positions: 'beginning' and 'end'; defaults to 'beginning' for the 'line' stack, 'end' for the others;
        // $this->crud->addButton($stack, $name, $type, $content, $position); // add a button; possible types are: view, model_function
        // $this->crud->addButtonFromModelFunction($stack, $name, $model_function_name, $position); // add a button whose HTML is returned by a method in the CRUD model
        $this->crud->addButtonFromView('line', 'actions_claim', 'actions_claim'); // add a button whose HTML is in a view placed at resources\views\vendor\backpack\crud\buttons
        // $this->crud->removeButton($name);
        // $this->crud->removeButtonFromStack($name, $stack);
        // $this->crud->removeAllButtons();
        // $this->crud->removeAllButtonsFromStack('line');

        // ------ CRUD ACCESS
        // $this->crud->allowAccess(['list', 'create', 'update', 'reorder', 'delete']);
        // $this->crud->denyAccess(['list', 'create', 'update', 'reorder', 'delete']);
        $this->crud->denyAccess(['create', 'update', 'delete']);
        // ------ CRUD REORDER
        // $this->crud->enableReorder('label_name', MAX_TREE_LEVEL);
        // NOTE: you also need to do allow access to the right users: $this->crud->allowAccess('reorder');

        // ------ CRUD DETAILS ROW
        // $this->crud->enableDetailsRow();
        // NOTE: you also need to do allow access to the right users: $this->crud->allowAccess('details_row');
        // NOTE: you also need to do overwrite the showDetailsRow($id) method in your EntityCrudController to show whatever you'd like in the details row OR overwrite the views/backpack/crud/details_row.blade.php

        // ------ REVISIONS
        // You also need to use \Venturecraft\Revisionable\RevisionableTrait;
        // Please check out: https://laravel-backpack.readme.io/docs/crud#revisions
        // $this->crud->allowAccess('revisions');

        // ------ AJAX TABLE VIEW
        // Please note the drawbacks of this though:
        // - 1-n and n-n columns are not searchable
        // - date and datetime columns won't be sortable anymore
        // $this->crud->enableAjaxTable();

        // ------ DATATABLE EXPORT BUTTONS
        // Show export to PDF, CSV, XLS and Print buttons on the table view.
        // Does not work well with AJAX datatables.
        // $this->crud->enableExportButtons();

        // ------ ADVANCED QUERIES
        // $this->crud->addClause('active');
        // $this->crud->addClause('type', 'car');
        // $this->crud->addClause('where', 'name', '==', 'car');
        // $this->crud->addClause('whereName', 'car');
        // $this->crud->addClause('whereHas', 'posts', function($query) {
        //     $query->activePosts();
        // });
        // $this->crud->addClause('withoutGlobalScopes');
        // $this->crud->addClause('withoutGlobalScope', VisibleScope::class);
        // $this->crud->with(); // eager load relationships
        // $this->crud->orderBy();
        // $this->crud->groupBy();
        // $this->crud->limit();
        $this->crud->orderBy('id', 'DESC');

        $this->crud->setListView('crud::claims.list');
    }

    public function store(StoreRequest $request) {
        // your additional operations before save here
        $redirect_location = parent::storeCrud($request);
        // your additional operations after save here
        // use $this->data['entry'] or $this->crud->entry
        return $redirect_location;
    }

    public function update(UpdateRequest $request) {
        // your additional operations before save here
        $redirect_location = parent::updateCrud($request);
        // your additional operations after save here
        // use $this->data['entry'] or $this->crud->entry
        return $redirect_location;
    }

    public function actionClaim($id, $statusId) {
        $claim = $this->crud->getModel()->findOrFail($id);

        $claim->status = $statusId;

        if ($claim->save()) {
            $statusText = trans('claims.statuses.' . Claim::claimStatuses()[$statusId]);
            $message = trans('claims.notifications.status_change') . ": " . $statusText;
            $user = User::where(['id' => $claim->user_id])->first();

            if ($user) {
                $arUser = User::where(['id' => $claim->user_id])->get()->pluck('id')->toArray();
                $push_tokens = (new User)->getPushTokens($arUser, true);

                foreach ($push_tokens as $os_id => $os_tokens) {
                    if($os_id) {
                        // is iOS
                        Notification::send($os_tokens, new ClaimStatusChangingDefault($message));
                    } else {
                        // is Android
                        Notification::send($os_tokens, new ClaimStatusChanging($message));
                    }
                }
            }

            return [
                'result' => $statusText,
                'statusId' => $statusId,
            ];
        }
        return response()->json([], 503);
    }

    /**
     * @param $id
     *
     * @return array
     */
    public function acceptClaim($id) {
        return $this->actionClaim($id, 1);
    }

    /**
     * @param $id
     *
     * @return array
     */
    public function rejectClaim($id) {
        return $this->actionClaim($id, 2);
    }

    /**
     * @param $id
     *
     * @return array
     */
    public function closeClaim($id) {
        return $this->actionClaim($id, 3);
    }

    /**
     * Display all rows in the database for this entity.
     *
     * @return Response
     */
    public function index()
    {
        $this->crud->hasAccessOrFail('list');

        if ($this->setCookie('admin/claim', 'true')) {
            return redirect(request()->url());
        }

        $this->data['crud'] = $this->crud;
        $this->data['title'] = ucfirst($this->crud->entity_name_plural);

        // get all entries if AJAX is not enabled
        if (! $this->data['crud']->ajaxTable()) {
            $this->data['entries'] = $this->data['crud']->getEntries();
        }

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getListView(), $this->data);
    }

    /**
     * @param \Illuminate\Http\Request $request
     *
     * @return array|\Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector|\Illuminate\View\View|mixed|void
     * @throws \Spatie\Permission\Exceptions\PermissionDoesNotExist
     */
    public function groupView(Request $request) {
        $this->crud->hasAccessOrFail('list');

        if ($this->setCookie('admin/claim', 'false')) {
            return redirect(request()->url());
        }

        $isFilterApplied = (bool) $request->get('apartment');
        if ($isFilterApplied) {
            $this->crud->addClause('where', 'apartment_id', '=', $request->get('apartment'));

            // Отключить сортировку по квартире при выбранном фильтре
            $this->crud->columns['apartment_number']['orderable'] = false;
        }

        $this->data['crud'] = $this->crud;

        // build menu tree
        $this->data['tree'] = (new Apartment)->getHouseTreeById();

        $this->data['badgeNotViewed'] = [];

        $fields = [
            ['name' => 'number', 'group' => '`number`'],
            ['name' => 'entrance', 'group' => '`entrance`'],
            ['name' => 'floor', 'group' => 'CONCAT(`floor`, "-", `entrance`)'],
        ];
        $fieldsStr = implode('`, `', array_column($fields, 'name'));

        foreach ($fields as $field) {
            $groupBy = $field['group'];
            $name = $field['name'];

            if (!Auth::user()->hasPermissionTo(Permission::FULL_ACCESS)) {
                $userclaimTypes = auth()->user()->claimTypes();
                if ($userclaimTypes->count() > 0) {

                    $types = [];
                    foreach ($userclaimTypes->get()->toArray() as $type) {
                        $types[] = $type['id'];
                    }

                    $typeCondition = '`claims`.`type_id` IN (' . implode(', ', $types) . ')';
                } else {
                    $typeCondition = '0 = 1';
                }
            } else {
                $typeCondition = '1 = 1';
            }

            $query =
                "
                SELECT 
                  count(*) as `count`, `$fieldsStr`
                FROM (
                  SELECT 
                    `claims`.`status`, 
                    `apartments`.`floor`, 
                    `apartments`.`id` AS `number`, 
                    `apartments`.`entrance` 
                  FROM `claims` 
                  LEFT JOIN `apartments` 
                  ON `claims`.`apartment_id` = `apartments`.`id` and `apartments`.`object_id` = '" . config('app.object.id') . "' 
                                                                 and `claims`.`object_id` = '" . config('app.object.id') . "' 
                  WHERE (`claims`.`status` = 0) AND ($typeCondition)
                ) `claims_not_viewed` 
                GROUP BY $groupBy
                ";

            $dbResult = DB::select($query);
            $result = [];

            if ($name === 'floor') {
                foreach ($dbResult as $item) {
                    $result[$item->entrance][$item->floor] = $item->count;
                }
            }
            else {
                foreach ($dbResult as $item) {
                    $result[$item->{$name}] = $item->count;
                }
            }

            $this->data['badgeNotViewed'][$name] = $result;

        }

        $this->data['selected'] = $request->all();

        if ($apartment = $request->get('apartment')) {
            $apartment = Apartment::find($apartment);

            if ($apartment instanceof Apartment) {
                $this->data['selected']['apartment'] = $apartment->getAttribute('number');
            }
        }

        // get all entries if AJAX is not enabled
        // always show all entries on group view if filter is applied
        if ($isFilterApplied || !$this->data['crud']->ajaxTable()) {
            $this->data['entries'] = $this->data['crud']->getEntries();
        }

        return view('crud::claims.group', $this->data);
    }

    public function show($id){
        abort(404);
    }


    // Перегрузка метода для поиска и выделения жирным активных записей
    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        // Замыкание для отображения выбранной квартиры на странице группировки
        $apartment = Apartment::where('number', $this->request->get('apartment'))
            ->where('object_id', config('app.object.id'))
            ->first();

        if ($apartment instanceof Apartment) {
            $this->crud->addClause('where', 'apartment_id', '=', $apartment->id);
            $this->crud->addClause('where', 'object_id', '=', config('app.object.id'));
        }

        /*
         * Стандартный поиск
         * Дубликат необходим для сортировки по "нестандартным" столбцам, которых нет в базе данных
         */
        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // overwrite any order set in the setup() method with the datatables order
        $sortedEntries = null;
        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);
            if ($column['tableColumn']) {
                // clear any past orderBy rules
                $this->crud->query->getQuery()->orders = null;
                // apply the current orderBy rules
                if(in_array($column['name'], ['claim_text'])) {
                    // для поля с именем number установлена отдельная сортировка для более удобочитаемого отображения квартир, имеющих номера
                    $this->crud->query = $this->crud->query->orderByRaw(
                        $column['name'] . '+0 '.$column_direction.', '.
                        $column['name'].' '.$column_direction.
                        ', LENGTH('.$column['name'].') '.$column_direction
                    );
                } else {
                    $this->crud->orderBy($column['name'], $column_direction);
                }
            } else {
                // Сортировка для полей, не входящих в состав таблицы
                if ($column['name'] === 'apartment_number') {
                    $sortedEntries = $this->sortByApartmentNumber();
                } elseif ($column['name'] === 'type') {
                    $sortedEntries = $this->sortByType();
                }
            }
        }

        /*
         * Конец стандартного поиска
         */
        $entries = $sortedEntries? $sortedEntries : $this->crud->getEntries();

        $data = $this->crud->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);

        /* Сформировать массив идентификаторов, которые будут выделяться жирным
         * isNew - которые будут выделяться жирным
         * inWork - которые будут выделяться очень жирным
         */
        $isNew = collect();
        $inWork = collect();

        for ($i = 0; $i < count($entries); $i++) {
            if ($entries[$i]->status === 0) {
                $isNew->push($i);
            } elseif ($entries[$i]->status === 1) {
                $inWork->push($i);
            }
        }

        foreach ($isNew as $newId) {
            $data['data'][$newId]['DT_RowClass'] = 'font-weight-900';
        }

        foreach ($inWork as $workId) {
            $data['data'][$workId]['DT_RowClass'] = 'font-weight-600';
        }
        
        // Добавить идентификатор для каждого tr
        $i = 0;
        foreach($entries as $entry) {
            $data['data'][$i]['DT_RowAttr'] = ['data-entry-id' => $entry->id];
            $i++;
        }

        return $data;
    }

    /**
     * Сортировка по номеру квартиры
     *
     * @return \Illuminate\Support\Collection
     */
    protected function sortByApartmentNumber()
    {
        // Получение настроек
        list($offset, $limit, $direction) = $this->getOrderSettings();

        // Взять все заявки из базы данных с присоединением стоблца номеров
        $raw_claims_builder = DB::table('claims')
            ->where('claims.object_id', config('app.object.id'))
            ->join('apartments', 'apartments.id', '=', 'claims.apartment_id')
            ->select('apartments.number', 'claims.*');

        // Проверка входящих значений limit, offset и direction
        $valid_settings = $this->validateOrderSettings($offset, $limit, $direction);

        if ($valid_settings['offset']) {
            $raw_claims_builder->offset($offset);
        }

        if ($valid_settings['limit']) {
            $raw_claims_builder->limit($limit);
        }

        if (!$valid_settings['direction']) {
            $direction = 'acs';
        }

        // Выполнить "натуральную" сортировку по номеру квартиры, выбрав направление
        $raw_claims_builder->orderByRaw(
            'apartments.number+0 '.$direction.','.
            'apartments.number '.$direction.','.
            'LENGTH(\'apartments.number\')' . $direction
        );

        $raw_claims = $raw_claims_builder->get();

        return $this->createOrderedClaims($raw_claims_builder->get());
    }

    protected function sortByType()
    {
        // Получение настроек
        list($offset, $limit, $direction) = $this->getOrderSettings();

        // Получить все заявки из таблицы типов
        $raw_type_builder = DB::table('claims')
            ->where('claims.object_id', config('app.object.id'))
            ->join('claim_type', 'claim_type.id', '=', 'claims.type_id')
            ->select('claims.*', 'claim_type.text');

        // Проверка входящих значений limit и offset
        $valid_settings = $this->validateOrderSettings($offset, $limit, $direction);

        if ($valid_settings['offset']) {
            $raw_type_builder->offset($offset);
        }

        if ($valid_settings['limit']) {
            $raw_type_builder->limit($limit);
        }

        if (!$valid_settings['direction']) {
            $direction = 'acs';
        }

        $isFilterApplied = (bool) $this->request->get('apartment');
        if ($isFilterApplied) {

            // Получить квартиру по ее номеру
            $apartment_id = Apartment::where('number', $this->request->get('apartment'))
                ->where('object_id', config('app.object.id'))
                ->first()
                ->id;

            $raw_type_builder
                ->where('apartment_id', $apartment_id);
        }

        $raw_type_builder->orderBy('claim_type.text', $direction);

        return $this->createOrderedClaims($raw_type_builder->get());
    }

    /**
     * Получение настроек для сортировки
     *
     * @return array
     */
    private function getOrderSettings()
    {
        $offset = $this->request->start;
        $limit = $this->request->length;
        $direction = $this->request->order[0]['dir'];

        return [$offset, $limit, $direction];
    }

    /**
     * Валидация настроек сортировки
     *
     * @param null $offset
     * @param null $limit
     * @param null $direction
     * @return \Illuminate\Support\Collection
     */
    private function validateOrderSettings($offset = null, $limit = null, $direction = null)
    {
        $status = collect([
            'offset'        => false,
            'limit'         => false,
            'direction'     => false
        ]);

        // Проверка входящиз значений limit, offset и direction
        $int_reg = '{^[0-9]+$}';
        if (preg_match($int_reg, $offset) && $offset > 0) {
            $status->put('offset', true);
        }

        if ((preg_match($int_reg, $limit) && $limit > 0)) {
            $status->put('limit', true);
        }

        // Проверка входящего значения direction
        if ((preg_match('{^asc|desc$}', $direction))) {
            $status->put('direction', true);
        }

        return $status;
    }

    /**
     * Создать заявки
     *
     * @param $raw_claims
     * @return \Illuminate\Support\Collection
     */
    private function createOrderedClaims($raw_claims)
    {
        // На основе полученных данных создать заявки
        $claims = collect();
        foreach ($raw_claims as $raw_claim) {
            $claim = new Claim();

            foreach ($raw_claim as $attribute => $value) {
                $claim->$attribute = $value;
            }

            $claims->push($claim);
        }

        return $claims;
    }
}
