<?php

namespace App\Http\Controllers\Admin;

use App\Models\CallingPanel;
use App\Models\PermissionManager\Permission;
use App\Traits\Admin\Ordering;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\CRUD\app\Http\Controllers\CrudController;
use Illuminate\Support\Facades\Auth;

// VALIDATION: change the requests to match your own file names if you need form validation
use App\Http\Requests\CallingPanelRequest as StoreRequest;
use App\Http\Requests\CallingPanelRequest as UpdateRequest;

class CallingPanelCrudController extends CrudController
{
    use CheckingPermissions, Ordering;

    /**
     * CallingPanelCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::CALLING_PANELS_CREATE_ACCESS],
            ['create', 'store'],
        ], [
            [Permission::CALLING_PANELS_EDIT_ACCESS, Permission::CALLING_PANELS_VIEW_LIMITED_ACCESS],
            ['index'],
        ], [
            [Permission::CALLING_PANELS_EDIT_ACCESS],
            ['edit', 'update'],
        ], [
            [Permission::CALLING_PANELS_DELETE_ACCESS],
            ['destroy']
        ]]);

    }

    public function setup()
    {
        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\CallingPanel');
        $this->crud->setRoute(config('backpack.base.route_prefix') . '/calling_panel');
        $this->crud->setEntityNameStrings(__('calling_panels.calling_panel_singular'), __('calling_panels.calling_panel_plural'));

        $this->applyBackpackPermissions();

        $this->crud->setCreateView('crud::calling_panels.create');
        $this->crud->setEditView('crud::calling_panels.edit');

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */

        //$this->crud->setFromDb();

        // ------ CRUD FIELDS

        $this->crud->addField([
            'name' => 'name',
            'label' => __("calling_panels.crud_fields.name"),
            'type' => 'text',
        ]);

        $this->crud->addField([
            'name' => 'sip_number',
            'label' => __("calling_panels.crud_fields.sip_number"),
            'type' => 'text',
        ]);

        $this->crud->addField([
            'name'  => 'rtsp_link_type',
            'label' => __("calling_panels.crud_fields.rtsp_link_type"),
            'type'  => 'radio',
            'options' => [
                CallingPanel::RTSP_LINK_TYPE_FULL => __('validation.values.rtsp_link_type.' . CallingPanel::RTSP_LINK_TYPE_FULL),
                CallingPanel::RTSP_LINK_TYPE_PARAMETERS => __('validation.values.rtsp_link_type.' . CallingPanel::RTSP_LINK_TYPE_PARAMETERS),
            ],
            'inline' => true,
            'default' => CallingPanel::RTSP_LINK_TYPE_FULL,
        ])->afterField('sip_number');

        $this->crud->addField([
            'name' => 'rtsp_link',
            'label' => __("calling_panels.crud_fields.rtsp_link"),
            'type' => 'url',
        ]);

        $this->crud->addField([
            'name' => 'login',
            'label' => __("calling_panels.crud_fields.login"),
            'type' => 'text',
        ]);

        $this->crud->addField([
            'name' => 'password',
            'label' => __("calling_panels.crud_fields.password"),
            'type' => 'text',
        ]);

        $this->crud->addField([
            'name' => 'ip_address',
            'label' => __("calling_panels.crud_fields.ip_address"),
            'type' => 'text',
        ]);

        $this->crud->addField([
            'name' => 'port',
            'label' => __("calling_panels.crud_fields.port"),
            'type' => 'text',
        ]);

        $this->crud->addField([
            'name' => 'dtmf_1',
            'label' => __("calling_panels.crud_fields.dtmf_1"),
            'type' => 'text',
        ]);

        $this->crud->addField([
            'name'  => 'has_second_relay',
            'label' => __("calling_panels.crud_fields.has_second_relay"),
            'type'  => 'radio',
            'options' => [
                0 => __('validation.values.has_second_relay.0'),
                1 => __('validation.values.has_second_relay.1'),
            ],
            'inline' => true,
            'default' => 0,
        ])->afterField('dtmf_1');

        $this->crud->addField([
            'name' => 'dtmf_2',
            'label' => __("calling_panels.crud_fields.dtmf_2"),
            'type' => 'text',
        ]);

        $this->crud->addField([
            'name' => 'object_id',
            'type' => 'hidden',
            'value' => config('app.object.id')
        ]);

        $this->crud->addField([
            'name'  => 'is_callable',
            'label' => __("calling_panels.crud_fields.is_callable"),
            'type'  => 'checkbox',
            'options' => [
                0 => __('validation.values.has_second_relay.0'),
                1 => __('validation.values.has_second_relay.1'),
            ],
            'inline' => true,
            'default' => 1,
        ])->afterField('object_id');


        $this->crud->addField([
            'label' => __('apartments.crud_fields.binding_to_apartments'),
            'type' => 'apartments_checklist',
            'name' => 'apartments',
            'entity' => 'apartments',
            'attributes' => [
                'number' => __('apartments.apartment')
            ],
            'object_id' => config('app.object.id'),
            'model' => "\App\Models\Apartment",
            'pivot' => true,
            'check_all' => false,
        ], 'update')->afterField('is_callable');

        $this->crud->addField([
            'label' => __('apartments.crud_fields.binding_to_apartments'),
            'type' => 'apartments_checklist',
            'name' => 'apartments',
            'entity' => 'apartments',
            'attributes' => [
                'number' => __('apartments.apartment')
            ],
            'object_id' => config('app.object.id'),
            'model' => "\App\Models\Apartment",
            'pivot' => true,
            'check_all' => true,
        ], 'create')->afterField('is_callable');

//        foreach ($this->crud->create_fields as $index => $field) {
//            $this->crud->create_fields[$index]['label']
//                = $this->crud->update_fields[$index]['label']
//                = __("calling_panels.crud_fields.$index");
//        }

        // $this->crud->addField($options, 'update/create/both');
        // $this->crud->addFields($array_of_arrays, 'update/create/both');
        // $this->crud->removeField('name', 'update/create/both');
        // $this->crud->removeFields($array_of_names, 'update/create/both');
        //dd('$this->crud->create_fields:', $this->crud->create_fields);

        // ------ CRUD COLUMNS
        $this->crud->addColumn([
            'name' => 'name',
            'label' => __("calling_panels.crud_fields.name"),
            'type' => 'text',
        ]);

        $this->crud->addColumn([
            'name' => 'sip_number',
            'label' => __("calling_panels.crud_fields.sip_number"),
            'type' => 'text',
        ]);

        if (\Auth::user()->can(Permission::CALLING_PANELS_EDIT_ACCESS)) {
            $this->crud->addColumn([
                'name' => 'rtsp_link',
                'label' => __("calling_panels.crud_fields.rtsp_link"),
                'type' => 'model_function',
                'function_name' => 'getRtsp'
            ]);

            $this->crud->addColumn([
                'name' => 'dtmf_1',
                'label' => __("calling_panels.crud_fields.dtmf_1"),
                'type' => 'text',
            ]);

            $this->crud->addColumn([
                'name' => 'dtmf_2',
                'label' => __("calling_panels.crud_fields.dtmf_2"),
                'type' => 'text',
            ]);

            $this->crud->addColumn([
                'name' => 'is_callable',
                'label' => __("calling_panels.crud_fields.is_callable"),
                'type' => 'boolean',
            ]);
        } else {
            $this->crud->removeAllButtonsFromStack('line');
        }

        $this->setButtons();

        $this->crud->addClause('where', 'object_id', '=', config('app.object.id'));


        // $this->crud->addColumn(); // add a single column, at the end of the stack
        // $this->crud->addColumns(); // add multiple columns, at the end of the stack
        // $this->crud->removeColumn('column_name'); // remove a column from the stack
        // $this->crud->removeColumns(['column_name_1', 'column_name_2']); // remove an array of columns from the stack
        // $this->crud->setColumnDetails('column_name', ['attribute' => 'value']); // adjusts the properties of the passed in column (by name)
        // $this->crud->setColumnsDetails(['column_1', 'column_2'], ['attribute' => 'value']);

        // ------ CRUD BUTTONS
        // possible positions: 'beginning' and 'end'; defaults to 'beginning' for the 'line' stack, 'end' for the others;
        // $this->crud->addButton($stack, $name, $type, $content, $position); // add a button; possible types are: view, model_function
        // $this->crud->addButtonFromModelFunction($stack, $name, $model_function_name, $position); // add a button whose HTML is returned by a method in the CRUD model
        // $this->crud->addButtonFromView($stack, $name, $view, $position); // add a button whose HTML is in a view placed at resources\views\vendor\backpack\crud\buttons
        // $this->crud->removeButton($name);
        // $this->crud->removeButtonFromStack($name, $stack);
        // $this->crud->removeAllButtons();
        // $this->crud->removeAllButtonsFromStack('line');

        // ------ CRUD ACCESS
        // $this->crud->allowAccess(['list', 'create', 'update', 'reorder', 'delete']);
        // $this->crud->denyAccess(['list', 'create', 'update', 'reorder', 'delete']);

        // ------ CRUD REORDER
        // $this->crud->enableReorder('label_name', MAX_TREE_LEVEL);
        // NOTE: you also need to do allow access to the right users: $this->crud->allowAccess('reorder');

        // ------ CRUD DETAILS ROW
        // $this->crud->enableDetailsRow();
        // NOTE: you also need to do allow access to the right users: $this->crud->allowAccess('details_row');
        // NOTE: you also need to do overwrite the showDetailsRow($id) method in your EntityCrudController to show whatever you'd like in the details row OR overwrite the views/backpack/crud/details_row.blade.php

        // ------ REVISIONS
        // You also need to use \Venturecraft\Revisionable\RevisionableTrait;
        // Please check out: https://laravel-backpack.readme.io/docs/crud#revisions
        // $this->crud->allowAccess('revisions');

        // ------ AJAX TABLE VIEW
        // Please note the drawbacks of this though:
        // - 1-n and n-n columns are not searchable
        // - date and datetime columns won't be sortable anymore
        // $this->crud->enableAjaxTable();

        // ------ DATATABLE EXPORT BUTTONS
        // Show export to PDF, CSV, XLS and Print buttons on the table view.
        // Does not work well with AJAX datatables.
        // $this->crud->enableExportButtons();

        // ------ ADVANCED QUERIES
        // $this->crud->addClause('active');
        // $this->crud->addClause('type', 'car');
        // $this->crud->addClause('where', 'name', '==', 'car');
        // $this->crud->addClause('whereName', 'car');
        // $this->crud->addClause('whereHas', 'posts', function($query) {
        //     $query->activePosts();
        // });
        // $this->crud->addClause('withoutGlobalScopes');
        // $this->crud->addClause('withoutGlobalScope', VisibleScope::class);
        // $this->crud->with(); // eager load relationships
        // $this->crud->orderBy();
        // $this->crud->groupBy();
        // $this->crud->limit();
    }

    public function store(StoreRequest $request)
    {
        $request->preProcess();

        // your additional operations before save here
        $redirect_location = parent::storeCrud($request);
        // your additional operations after save here
        // use $this->data['entry'] or $this->crud->entry
        return $redirect_location;
    }

    public function update(UpdateRequest $request)
    {
        $request->preProcess();

        // your additional operations before save here
        $redirect_location = parent::updateCrud($request);
        // your additional operations after save here
        // use $this->data['entry'] or $this->crud->entry
        return $redirect_location;
    }

    public function edit($id)
    {
        $this->crud->hasAccessOrFail('update');

        // get the info for that entry
        $this->data['entry'] = $this->crud->getEntry($id);
        // get the info for that entry
        $this->data['entry'] = $this->crud->getEntry($id);
        if($this->data['entry']->object_id  != config('app.object.id')){
            return \Redirect::to($this->crud->route);
        }
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['title'] = trans('backpack::crud.edit').' '.$this->crud->entity_name;

        $this->data['id'] = $id;

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getEditView(), $this->data);
    }

    protected function setButtons()
    {
        $this->crud->removeAllButtons();

        if (Auth::user()->can(Permission::CALLING_PANELS_CREATE_ACCESS)) {
            $this->crud->addButton('top', 'create', 'view', 'crud::buttons.create');
        }
        if (Auth::user()->can(Permission::CALLING_PANELS_DELETE_ACCESS)) {
            $this->crud->addButton('line', 'delete', 'view', 'crud::buttons.delete');
        }
        if (Auth::user()->can(Permission::CALLING_PANELS_EDIT_ACCESS)) {
            $this->crud->addButton('line', 'update', 'view', 'crud::buttons.update');
        }
    }

    /**
     * Get the save configured save action or the one stored in a session variable.
     * @return [type] [description]
     */
    public function getSaveAction()
    {
        $saveAction = session('save_action', config('backpack.crud.default_save_action', 'save_and_back'));
        $saveOptions = [];
        $saveCurrent = [
            'value' => $saveAction,
            'label' => $this->getSaveActionButtonName($saveAction),
        ];

        switch ($saveAction) {
            case 'save_and_edit':
                $saveOptions['save_and_back'] = $this->getSaveActionButtonName('save_and_back');
                if (Auth::user()->can(Permission::CALLING_PANELS_CREATE_ACCESS)) {
                    $saveOptions['save_and_new'] = $this->getSaveActionButtonName('save_and_new');
                }
                break;
            case 'save_and_new':
                if (Auth::user()->can(Permission::CALLING_PANELS_CREATE_ACCESS)) {
                    $saveOptions['save_and_back'] = $this->getSaveActionButtonName('save_and_back');
                }
                if (Auth::user()->can(Permission::CALLING_PANELS_EDIT_ACCESS)) {
                    $saveOptions['save_and_edit'] = $this->getSaveActionButtonName('save_and_edit');
                }
                break;
            case 'save_and_back':
            default:
                if (Auth::user()->can(Permission::CALLING_PANELS_EDIT_ACCESS)) {
                    $saveOptions['save_and_edit'] = $this->getSaveActionButtonName('save_and_edit');
                }
                if (Auth::user()->can(Permission::CALLING_PANELS_CREATE_ACCESS)) {
                    $saveOptions['save_and_new'] = $this->getSaveActionButtonName('save_and_new');
                }
                break;
        }

        return [
            'active' => $saveCurrent,
            'options' => $saveOptions,
        ];
    }

    /**
     * Get the translated text for the Save button.
     * @param  string $actionValue [description]
     * @return [type]              [description]
     */
    private function getSaveActionButtonName($actionValue = 'save_and_back')
    {
        switch ($actionValue) {
            case 'save_and_edit':
                return trans('backpack::crud.save_action_save_and_edit');
                break;
            case 'save_and_new':
                return trans('backpack::crud.save_action_save_and_new');
                break;
            case 'save_and_back':
            default:
                return trans('backpack::crud.save_action_save_and_back');
                break;
        }
    }

    public function show($id){
        Abort(404);
    }

    /**
     * Переопределение необходимо для сортировки ненастраиваемых полей
     *
     * @return array|\Backpack\CRUD\app\Http\Controllers\Operations\JSON
     * @throws \Backpack\CRUD\Exception\AccessDeniedException
     */
    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // overwrite any order set in the setup() method with the datatables order
        $rawSortedEntries = null;
        $sortedEntries = collect();
        if ($this->request->input('order')) {
            //dd(1);
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);

            $customOrder = true;
            $table = $this->crud->getModel()->getTable();
            switch ($column['name']) {
                // По столбцу название
                case 'name':
                    $rawSortedEntries = $this->sortBySimpleRow($table, 'name');
                    break;
                case 'sip_number':
                    $rawSortedEntries = $this->sortBySimpleRow($table, 'sip_number');
                    break;
                case 'rtsp_link':
                    $rawSortedEntries = $this->sortBySimpleRow($table, 'rtsp_link', [
                        'type' => 'length'
                    ]);
                    break;
                case 'dtmf_1':
                    $rawSortedEntries = $this->sortBySimpleRow($table, 'dtmf_1');
                    break;
                case 'dtmf_2':
                    $rawSortedEntries = $this->sortBySimpleRow($table, 'dtmf_2');
                    break;
                case 'is_callable':
                    $rawSortedEntries = $this->sortBySimpleRow($table, 'is_callable_string', [
                        'select' => '*, IF(is_callable = 1,\''.__('crud.yes').'\',\''.__('crud.no').'\') AS is_callable_string'
                    ]);
                    break;
                default:
                    $customOrder = false;
                    if ($column['tableColumn']) {
                        // clear any past orderBy rules
                        $this->crud->query->getQuery()->orders = null;
                        // apply the current orderBy rules
                        $this->crud->orderBy($column['name'], $column_direction);
                    }
                    break;
            }

            // Если сортировка кастомная, то сформировать коллекцию объектов
            if ($customOrder) {
                $sortedEntries = $this->createOrderedObjectsByIds('App\Models\CallingPanel', $rawSortedEntries);
            }
        }

        // Если сортировка кастомная, то присваиваем отсортированные записи
        $entries = $sortedEntries->count()? $sortedEntries : $this->crud->getEntries();

        return $this->crud->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);
    }
}
