<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\AcsHttpRequestHelper;
use App\Models\Building;
use App\Models\PermissionManager\Permission;
use App\Traits\Admin\Ordering;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\CRUD\app\Http\Controllers\CrudController;
use Illuminate\Support\Facades\App;

// VALIDATION: change the requests to match your own file names if you need form validation
use App\Http\Requests\BuildingRequest as StoreRequest;
use App\Http\Requests\BuildingRequest as UpdateRequest;
use Illuminate\Http\Request as StoreCrudRequest;
use Illuminate\Http\Request as UpdateCrudRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class BuildingCrudController extends CrudController
{
    use CheckingPermissions, Ordering;

    /**
     * BuildingCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::FULL_ACCESS],
            ['create', 'store', 'index', 'edit', 'update', 'destroy'],
        ]]);
    }

    public function setup()
    {

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\Building');
        $this->crud->setRoute(config('backpack.base.route_prefix') . '/objects');
        $this->crud->setEntityNameStrings(__('objects.object_singular'), __('objects.objects_plural'));

        $this->applyBackpackPermissions();

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */

        $this->crud->addColumn([
            'name' => 'name',
            'label' => __('objects.crud_fields.name')
        ]);

        $this->crud->addColumn([
            'name' => 'address',
            'label' => __('objects.crud_fields.address')
        ]);

        $this->crud->addColumn([
            'name' => 'sip_server_address',
            'label' => __('objects.crud_fields.sip_server_address')
        ]);

        $this->crud->addColumn([
            'name' => 'sip_server_port',
            'label' => __('objects.crud_fields.sip_server_port')
        ]);

        $this->crud->addColumn([
            'name' => 'concierge_sip_number',
            'label' => __('objects.crud_fields.concierge_sip_number')
        ]);

//        $this->crud->addField([
//            'name' => 'name',
//            'label' => __('objects.crud_fields.name'),
//            'type' => 'multilang_text',
//            'langs' =>  ['ru', 'en']
//        ]);
//
//        $this->crud->addField([
//            'name' => 'address',
//            'label' => __('objects.crud_fields.address'),
//            'type' => 'multilang_text',
//            'langs' =>  ['ru', 'en']
//        ]);

        $this->crud->addField([
            'name' => 'name',
            'label' => __('objects.crud_fields.name'),
            'type' => 'text',
        ]);

        $this->crud->addField([
            'name' => 'address',
            'label' => __('objects.crud_fields.address'),
            'type' => 'text',
        ]);

        $this->crud->addField([
            'name' => 'sip_server_address',
            'label' => __('objects.crud_fields.sip_server_address'),
            'type' => 'text'
        ]);

        $this->crud->addField([
            'name' => 'sip_server_port',
            'label' => __('objects.crud_fields.sip_server_port'),
            'type' => 'text'
        ]);

        $this->crud->addField([
            'name' => 'concierge_sip_number',
            'label' => __('objects.crud_fields.concierge_sip_number'),
            'type' => 'text'
        ]);

        $this->crud->disableGroupedErrors();
        $this->crud->denyAccess(['delete']);

        $this->showUserObjects();
    }

    /**
     * Show only user objects
     */
    protected function showUserObjects()
    {
        // Get user objects list with rules
        $object_list = DB::table('permission_users')
            ->selectRaw('DISTINCT object_id')
            ->where('user_id', Auth::user()->id)
            ->pluck('object_id');

        $this->crud->addClause('whereIn', 'id', $object_list->toArray());
    }

    public function store(StoreRequest $request)
    {
        DB::beginTransaction();
        // your additional operations before save here
        $redirect_location = $this->storeCrud($request);
        $response = (new AcsHttpRequestHelper())->sendPost('/api/object', $this->data['entry']->toArray());
        switch ($response['status']) {
            case 200:
                DB::commit();
                // show a success message
                \Alert::success(trans('backpack::crud.insert_success'))->flash();
                // your additional operations before save here
                return $redirect_location;
                break;
            default:
                DB::rollBack();
                \Alert::error(__('response_error.' . $response['status']))->flash();
                return back()->withInput();
                break;
        }
    }

    public function update(UpdateRequest $request)
    {
        DB::beginTransaction();
        // your additional opera
        // your additional operations before save here
        $redirect_location = $this->updateCrud($request);
        $response = (new AcsHttpRequestHelper())->sendPut('/api/object/' . $this->data['entry']->id, $this->data['entry']->toArray());
        // your additional operations after save here
        // use $this->data['entry'] or $this->crud->entry
        switch ($response['status']) {
            case 200:
                DB::commit();
                // show a success message
                \Alert::success(trans('backpack::crud.insert_success'))->flash();
                // your additional operations before save here
                return $redirect_location;
                break;
            default:
                DB::rollBack();
                \Alert::error(__('response_error.' . $response['status']))->flash();
                return back()->withInput();
                break;
        }
    }

    /**
     * Store a newly created resource in the database.
     *
     * @param StoreRequest $request - type injection used for validation using Requests
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function storeCrud(StoreCrudRequest $request = null)
    {
        $this->crud->hasAccessOrFail('create');

        // fallback to global request instance
        if (is_null($request)) {
            $request = \Request::instance();
        }

        // replace empty values with NULL, so that it will work with MySQL strict mode on
        foreach ($request->input() as $key => $value) {
            if (empty($value) && $value !== '0') {
                $request->request->set($key, null);
            }
        }

        $model = $this->crud->getModel();
        $attributes = $request->except(['save_action', '_token', '_method']);

        foreach ($attributes as $key => $row) {
            if (in_array($key, $model->translatable)) {
                $array = [];

                foreach (config('app.locales') as $locale) {
                    $array = array_merge($array, [$locale => $row]);
                }

                $attributes[$key] = \GuzzleHttp\json_encode($array);
            }
        }

        $model->setRawAttributes($attributes);
        $model->save();

        $this->data['entry'] = $this->crud->entry = $model;

        // save the redirect choice for next time
        $this->setSaveAction();

        return $this->performSaveAction($model->getKey());
    }

    public function updateCrud(UpdateCrudRequest $request = null)
    {
        $this->crud->hasAccessOrFail('update');

        // fallback to global request instance
        if (is_null($request)) {
            $request = \Request::instance();
        }

        // replace empty values with NULL, so that it will work with MySQL strict mode on
        foreach ($request->input() as $key => $value) {
            if (empty($value) && $value !== '0') {
                $request->request->set($key, null);
            }
        }

        // update the row in the db
        $item = $this->crud->update($request->get($this->crud->model->getKeyName()),
            $request->except('save_action', '_token', '_method', 'current_tab'));
        $this->data['entry'] = $this->crud->entry = $item;

        // save the redirect choice for next time
        $this->setSaveAction();

        return $this->performSaveAction($item->getKey());
    }

    public function show($id)
    {
        Abort(404);
    }

    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // overwrite any order set in the setup() method with the datatables order
        $rawSortedEntries = null;
        $sortedEntries = collect();
        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);

            $customOrder = true;
            switch ($column['name']) {
                // По столбцу название
                case 'name':
                        $rawSortedEntries = $this->sortByName();
                    break;
                case 'address':
                    $rawSortedEntries = $this->sortByAddress();
                    break;
                case 'sip_server_address':
                    $rawSortedEntries = $this->sortBySimpleRows('sip_server_address');
                    break;
                case 'sip_server_port':
                    $rawSortedEntries = $this->sortBySimpleRows('sip_server_port');
                    break;
                case 'concierge_sip_number':
                    $rawSortedEntries = $this->sortBySimpleRows('concierge_sip_number');
                    break;
                default:
                    $customOrder = false;
                    if ($column['tableColumn']) {
                        // clear any past orderBy rules
                        $this->crud->query->getQuery()->orders = null;
                        // apply the current orderBy rules
                        $this->crud->orderBy($column['name'], $column_direction);
                    }
                    break;
            }

            // Если сортировка кастомная, то сформировать коллекцию объектов
            if ($customOrder) {
                $sortedEntries = $this->createOrderedObjects($rawSortedEntries);
            }
        }

        // Если сортировка кастомная, то присваиваем отсортированные записи
        $entries = $sortedEntries->count()? $sortedEntries : $this->crud->getEntries();

        return $this->crud->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);
    }

    /**
     * @param $raw_claims
     * @return \Illuminate\Support\Collection
     */
    private function createOrderedObjects($raw_objects)
    {
        // На основе полученных данных создать объекты
        $objects = collect();
        foreach ($raw_objects as $raw_object) {
            $object = new Building();
            $object->translatable = null;
            foreach ($raw_object as $attribute => $value) {
                if (in_array($attribute, ['name', 'address'])) {
                    $object->$attribute = mb_substr(mb_substr($value,0,mb_strlen($value)-1), 1);
                } else {
                    $object->$attribute = $value;
                }
            }

            $objects->push($object);
        }

        return $objects;
    }

    /**
     * Сортировка по названию объекта
     * @return \Illuminate\Support\Collection
     */
    protected function sortByName()
    {
        list($offset, $limit, $direction) = $this->getOrderSettings();

        $lang = App::getLocale();

        // Получить все объекты из таблицы объектов
        $row = 'JSON_EXTRACT(name,"$.'.$lang.'")';
        $raw_object_builder = DB::table('objects')
            ->selectRaw('*, '.$row .' AS name, JSON_EXTRACT(address, "$.'.$lang.'") AS address');

        // Проверка входящих значений limit и offset
        $valid_settings = $this->validateOrderSettings($offset, $limit, $direction);

        if ($valid_settings['offset']) {
            $raw_object_builder->offset($offset);
        }

        if ($valid_settings['limit']) {
            $raw_object_builder->limit($limit);
        }

        if (!$valid_settings['direction']) {
            $direction = 'acs';
        }

        // Начать сортировку
        $raw_object_builder->orderByRaw('
            LOWER('.$row.')+0 '.$direction.',
            LOWER('.$row.') '.$direction.',
            LENGTH('.$row.') '.$direction
        );

        return $raw_object_builder->get();
    }

    /**
     * Сортировка по адресу объекта
     * @return \Illuminate\Support\Collection
     */
    protected function sortByAddress()
    {
        list($offset, $limit, $direction) = $this->getOrderSettings();

        $lang = App::getLocale();

        // Получить все объекты из таблицы объектов
        $row = 'JSON_EXTRACT(address,"$.'.$lang.'")';
        $raw_object_builder = DB::table('objects')
            ->selectRaw('*, '.$row .' AS address, JSON_EXTRACT(name, "$.'.$lang.'") AS name');

        // Проверка входящих значений limit и offset
        $valid_settings = $this->validateOrderSettings($offset, $limit, $direction);

        if ($valid_settings['offset']) {
            $raw_object_builder->offset($offset);
        }

        if ($valid_settings['limit']) {
            $raw_object_builder->limit($limit);
        }

        if (!$valid_settings['direction']) {
            $direction = 'acs';
        }

        // Начать сортировку
        $raw_object_builder->orderByRaw('
            LOWER('.$row.')+0 '.$direction.',
            LOWER('.$row.') '.$direction.',
            LENGTH('.$row.') '.$direction
        );

        return $raw_object_builder->get();
    }


    protected function sortBySimpleRows($row)
    {
        list($offset, $limit, $direction) = $this->getOrderSettings();

        $lang = App::getLocale();

        // Получить все объекты из таблицы объектов
        $raw_object_builder = DB::table('objects')
            ->selectRaw('*, JSON_EXTRACT(name, "$.'.$lang.'") AS name, JSON_EXTRACT(address, "$.'.$lang.'") AS address');

        // Проверка входящих значений limit и offset
        $valid_settings = $this->validateOrderSettings($offset, $limit, $direction);

        if ($valid_settings['offset']) {
            $raw_object_builder->offset($offset);
        }

        if ($valid_settings['limit']) {
            $raw_object_builder->limit($limit);
        }

        if (!$valid_settings['direction']) {
            $direction = 'acs';
        }

        // Начать сортировку
        $raw_object_builder->orderByRaw('
            LOWER('.$row.')+0 '.$direction.',
            LOWER('.$row.') '.$direction.',
            LENGTH('.$row.') '.$direction
        );

        return $raw_object_builder->get();
    }
}
