<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\AcsHttpRequestHelper;
use App\Helpers\Objects;
use App\Models\Apartment;
use App\Http\Controllers\Controller;
use App\Models\ApartmentsGeneration;
use App\Traits\PermissionManager\CheckingPermissions;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Session;

class ApartmentGenerationController extends Controller
{
    use CheckingPermissions;

    public $total_available_activations = 0;
    public $request = null;
    public $save_numbers = 0;

    public function __construct() {
        $asc_helper = new AcsHttpRequestHelper();
        $response = $asc_helper->sendGet('/api/activations');

        if(!empty($response) && $response['status'] == 200) {
            if(isset($response['body']->used_activations_count)
                && isset($response['body']->available_activations_count)) {

                $this->total_available_activations = intval($response['body']->available_activations_count) - intval($response['body']->used_activations_count);

            }
        }

        $this->save_numbers = Apartment::where('is_local', 1)
            ->where('is_save', 1)
            ->where('object_id', config('app.object.id'))
            ->count();
    }

    protected function view($data = []) {
        $apartments = Apartment::getGroupByHouse();
        $common_data = [
            'apartments' => $apartments,
            'apartments_duplicates_items' => Session::get('apartments_duplicates_items'),
            'apartments_group_info' => Apartment::getGroupByHouseInfo($apartments),
            'total_available_activations' => $this->total_available_activations,
            'edited_apartments' => Session::get('edited_apartments'),
            'save_numbers' => $this->save_numbers
        ];

        return view('apartment.generation.form_content', array_merge($common_data, $data));
    }

    public function index(Request $request) {
        return $this->view(['request' => $request]);
    }

    public function create(Request $request) {
        $this->validate($request, [
            'number_entrances' => 'required|numeric|not_in:0',
            'number_floors' => 'required|numeric|not_in:0',
            'number_apartments_per_floors' => 'required|numeric|not_in:0',
            'activation_limit' => 'required|numeric',
        ],
        [
            'required' =>  trans('apartments.validate.required'),
            'numeric' =>  trans('apartments.validate.numeric'),
            'not_in' =>  trans('apartments.validate.not_in'),
        ]);

        $generate_params = [
            'number_entrances' => $request->post('number_entrances'),
            'number_floors' => $request->post('number_floors'),
            'number_apartments_per_floors' => $request->post('number_apartments_per_floors'),
            'activation_limit' => $request->post('activation_limit'),
        ];

        $this->generateApartments($generate_params);

        return $this->view(['request' => $request]);
    }

    public function save(Request $request) {
        $apartments = $request->post('apartments');
        $check_name_duplicates = [];
        $ajax_response = [
            'error' => false
        ];
        $arInsert = [];

        if(!empty($apartments)) {

            $check_name_duplicates = Apartment::checkNameDuplicatesInGrouping($apartments);

            if(empty($check_name_duplicates)) {
                Apartment::where('object_id', config('app.object.id'))->update([
                    'is_save' => 1
                ]);

                foreach ($apartments as $entrance_number => $entrance) {
                    if (!empty($entrance)) {
                        foreach ($entrance as $floor_number => $floor) {
                            if (!empty($floor)) {
                                foreach ($floor as $apartment_id => $apartment) {
                                    $database_apartment = Apartment::where('id', $apartment_id)->first();

                                    if (!empty($database_apartment)
                                        && isset($apartment['number'])
                                        && isset($apartment['activation_limit'])) {


                                        if($request->ajax()) {
                                            $update = $database_apartment->update([
                                                'number' => $apartment['number'],
                                                'activation_limit' => $apartment['activation_limit'],
                                                'is_local' => 0,
                                                'is_save' => 0,
                                            ]);
                                            if($update) {
                                                $arInsert[] = [
                                                    'object_id' => $database_apartment->object_id,
                                                    'ds_id' => $database_apartment->id,
                                                    'entrance' => $database_apartment->entrance,
                                                    'floor' => $database_apartment->floor,
                                                    'number' => $database_apartment->number,
                                                    'available_activations_count' => $database_apartment->activation_limit
                                                ];
                                            }

                                        } else {
                                            $database_apartment->update([
                                                'number' => $apartment['number'],
                                                'activation_limit' => $apartment['activation_limit'],
                                                'is_save' => 1
                                            ]);
                                        }

                                    } else {
                                        $ajax_response['error'] = true;
                                        $ajax_response['message'] = 'Apartment id=' . $apartment_id . ' does not exist';
                                    }
                                }
                            }
                        }
                    }
                }

                if(!$request->ajax()) {
                    Apartment::removeLocal();
                } else {
                    if (!empty($arInsert)) {
                        sleep(1);
                        $asc_helper = new AcsHttpRequestHelper();

                        $response = $asc_helper->sendPost('/api/apartment', $arInsert);

                        switch ($response['status']) {
                            case 200:
                            case 201:
                                // show a success message
                                Log::info('Apartment id: ' . $database_apartment->id . ' object_id: ' . trans('backpack::crud.insert_success'));
                                break;
                            default:
                                $database_apartment->delete();
                                Log::error(__('response_error.' . $response['status']));
                                break;
                        }
                    }
                }

            }  else {
                $ajax_response['error'] = true;
                $ajax_response['message'] = 'Duplicates';
                $ajax_response['duplicates'] = $check_name_duplicates;
            }
        } else {
            $ajax_response['error'] = true;
            $ajax_response['message'] = 'Empty apartments list';
        }

        if($request->ajax()) {
            return response()->json($ajax_response);
        }

        return redirect()->back()
            ->with('apartments_duplicates_items', $check_name_duplicates)
            ->with('edited_apartments', $apartments);
    }

    public function finish(Request $request) {
        Apartment::where('object_id', config('app.object.id'))->update([
            'is_save' => 0
        ]);

        // Очистка от удаленных
        Apartment::removeLocal();

        return response()->json(['error' => false]);
    }

    public function remove(Request $request) {
        $response = ['error' => false];

        if(!empty($request->id)) {
            $apartment = Apartment::where('id', $request->id)->first();
            if($apartment) $apartment->delete();
        } else {
            $response['error'] = true;
        }

        $this->setNumbers();
        return response()->json($response);
    }

    public function setNumbers() {
        $object_id = config('app.object.id');
        $apartments = Apartment::where('object_id', $object_id)->get();
        $i = 1;

        foreach($apartments as $apartment) {
            $apartment->update([
                'number' => $i
            ]);
            $i++;
        }
    }

    protected function generateApartments($params = []) {
        if(!empty($params)) {
            Apartment::removeLocal();
            $object_id = config('app.object.id');
            $apartment_number = 1;
            $database_apartment = [];

            for($entrance = 1; $entrance <= $params['number_entrances']; $entrance++) {
                for($floor = 1; $floor <= $params['number_floors']; $floor++) {
                    for($apartment = 1; $apartment <= $params['number_apartments_per_floors']; $apartment++) {

                        $database_apartment[] = [
                            'number' => $apartment_number,
                            'floor' => $floor,
                            'entrance' => $entrance,
                            'activation_limit' => $params['activation_limit'],
                            'object_id' => $object_id,
                            'is_local' => 1
                        ];

                        $apartment_number++;
                    }
                }
            }

            Apartment::insertOrUpdateApartments($database_apartment);
        }
    }

}