<?php

namespace App\Http\Controllers\Admin;

use App\Models\Apartment;
use App\Models\ApartmentSipNumber;
use App\Models\Notice;
use App\Models\PermissionManager\Permission;
use App\Models\Review;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\CRUD\app\Http\Controllers\CrudController;
use App\Helpers\AcsHttpRequestHelper;
use Illuminate\Support\Facades\DB;

// VALIDATION: change the requests to match your own file names if you need form validation
use App\Http\Requests\ApartmentRequest as StoreRequest;
use App\Http\Requests\ApartmentRequest as UpdateRequest;
use Illuminate\Http\Request as StoreCrudRequest;
use Illuminate\Http\Request as UpdateCrudRequest;
use Illuminate\Support\Facades\Auth;

class ApartmentCrudController extends CrudController
{
    use CheckingPermissions;

    /**
     * This parameter allows you to readily specify the entries in the length drop down select list
     * that DataTables shows when pagination is enabled.
     */
    const DEFAULT_PAGE_LENGTH_MENU = [10, 20, 50, 100];

    /**
     * Default value in the page's length drop down select list
     */
    const DEFAULT_PAGE_LENGTH = 20;


    /**
     * @var int[] $lengthMenu
     */
    public $lengthMenu = self::DEFAULT_PAGE_LENGTH_MENU;


    /**
     * ApartmentCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::APARTMENTS_CREATE_ACCESS],
            ['create', 'store'],
        ], [
            [Permission::APARTMENTS_VIEW_FULL_ACCESS],
            ['show'],
        ], [
            [Permission::APARTMENTS_VIEW_FULL_ACCESS, Permission::APARTMENTS_VIEW_LIMITED_ACCESS],
            ['index'],
        ], [
            [Permission::APARTMENTS_EDIT_ACCESS],
            ['edit', 'update'],
        ], [
            [Permission::FULL_ACCESS],
            ['resetActivations'],
        ]]);
    }

    public function setup()
    {
        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\Apartment');
        $this->crud->setRoute(config('backpack.base.route_prefix') . '/apartment');
        $this->crud->setEntityNameStrings(__('apartments.apartment_singular_3'), __('apartments.apartment_plural'));

        $this->applyBackpackPermissions();

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */

        // ------ CRUD FIELDS
        // $this->crud->addField($options, 'update/create/both');
        // $this->crud->addFields($array_of_arrays, 'update/create/both');
        // $this->crud->removeField('name', 'update/create/both');
        // $this->crud->removeFields($array_of_names, 'update/create/both');

        $entrances = range(1, config('house.entrances_count'));

        $this->crud->addField([
            'name' => 'entrance',
            'label' => __('apartments.crud_fields.entrance'),
            'type' => 'select2_from_array',
            'options' => array_combine($entrances, $entrances),
        ]);

        $floors = range(1, config('house.floors_count'));
        $this->crud->addField([
            'name' => 'floor',
            'label' => __('apartments.crud_fields.floor'),
            'type' => 'select2_from_array',
            'options' => array_combine($floors, $floors),
        ]);

        $this->crud->addField([
            'name' => 'number',
            'label' => __('apartments.crud_fields.number'),
            'type' => 'text',
        ]);

        $this->crud->addField([
            'name' => 'activation_code',
            'label' => __('apartments.crud_fields.activation_code'),
            'type' => 'custom_code',
            'function' => 'getPrepareCodeHtml',
            'attributes' => [
                'disabled' => 'disabled'
            ]
        ], 'update');

        $this->crud->addField([
            'name' => 'sip_numbers',
            'label' => __('apartments.crud_fields.sip_numbers'),
            'type' => 'sip_table',
            'columns' => [
                'sip_number' => __('apartments.crud_fields.sip_number'),
                'sip_password' => __('apartments.crud_fields.sip_password'),
                'token' => __('apartments.crud_fields.sip_activated')
            ],
            'entity_singular' => __('apartments.crud_fields.sip_number_add_button'),
            'max' => Apartment::getMaxSipNumbers(),
            'min' => 0
        ]);

        $this->crud->addField([
            'name' => 'object_id',
            'label' => __('apartments.crud_fields.object_id'),
            'type' => 'hidden',
            'value' => config('app.object.id')
        ]);

        $this->crud->addField([
            'label' => __('apartments.crud_fields.select_cameras'),
            'type' => 'checklist_by_object',
            'name' => 'cameras',
            'entity' => 'cameras',
            'attribute' => 'name',
            'object_id' => config('app.object.id'),
            'model' => "\App\Models\Camera",
            'pivot' => true,
            'check_all' => false,
        ], 'update');

        $this->crud->addField([
            'label' => __('apartments.crud_fields.select_cameras'),
            'type' => 'checklist_by_object',
            'name' => 'cameras',
            'entity' => 'cameras',
            'attribute' => 'name',
            'object_id' => config('app.object.id'),
            'model' => "\App\Models\Camera",
            'pivot' => true,
            'check_all' => true,
        ], 'create');

        $this->crud->addField([
            'label' => __('apartments.crud_fields.select_calling_panels'),
            'type' => 'checklist_by_object',
            'name' => 'calling_panels',
            'entity' => 'calling_panels',
            'attribute' => 'name',
            'object_id' => config('app.object.id'),
            'model' => "\App\Models\CallingPanel",
            'pivot' => true,
            'check_all' => false,
        ], 'update');

        $this->crud->addField([
            'label' => __('apartments.crud_fields.select_calling_panels'),
            'type' => 'checklist_by_object',
            'name' => 'calling_panels',
            'entity' => 'calling_panels',
            'attribute' => 'name',
            'object_id' => config('app.object.id'),
            'model' => "\App\Models\CallingPanel",
            'pivot' => true,
            'check_all' => true,
        ], 'create');

        $this->crud->addColumn([
            'name' => 'entrance',
            'label' => __('apartments.crud_fields.entrance'),
            'type' => 'text',
        ]);

        $this->crud->addColumn([
            'name' => 'floor',
            'label' => __('apartments.crud_fields.floor'),
            'type' => 'text',
        ]);

        $this->crud->addColumn([
            'name' => 'number',
            'label' => __('apartments.crud_fields.number'),
            'type' => 'text',
        ]);

        if (Auth::user()->can(Permission::APARTMENTS_VIEW_FULL_ACCESS)) {
            $this->crud->addColumn([
                'name' => 'activation_code_code',
                'label' => __('apartments.crud_fields.activation_code'),
                'type' => 'model_function',
                'function_name' => 'getPrepareCodeHtml',
                'limit' => 500
            ]);

            $this->crud->addColumn([
                'name' => 'sip_numbers',
                'label' => __('apartments.crud_fields.sip_numbers'),
                'type' => 'sip_numbers',
                'function_name' => 'getSipNumbersHtmlForTable',
            ]);

            $this->crud->addColumn([
                'name' => 'prepare_activation',
                'label' => __('apartments.crud_fields.activations'),
                'type' => 'model_function',
                'function_name' => 'getPrepareActivationHtml',
                'limit' => 500
            ]);

            $this->crud->addButtonFromView('line', 'reset_activations', 'reset_activations', 'end'); // add a button whose HTML is in a view placed at resources\views\vendor\backpack\crud\buttons
            $this->crud->addButtonFromView('line', 'reset_code', 'reset_code', 'end'); // add a button whose HTML is in a view placed at resources\views\vendor\backpack\crud\buttons
        } else {
            $this->crud->removeAllButtonsFromStack('line');
        }

        // ------ CRUD ACCESS
        // $this->crud->allowAccess(['list', 'create', 'update', 'reorder', 'delete']);
        $this->crud->denyAccess(['delete']);
        $this->crud->addClause('where', 'object_id', '=', config('app.object.id'));

        $this->crud->addFilter([ // select2 filter
            'name' => 'entrance',
            'type' => 'select2',
            'label' => __('apartments.crud_fields.entrance')
        ], function () {
            return Apartment::getObjectEntrances();
        }, function ($value) { // if the filter is active
            $this->crud->addClause('where', 'entrance', $value);
        });

        $this->crud->addFilter([ // select2 filter
            'name' => 'floor',
            'type' => 'select2',
            'label' => __('apartments.crud_fields.floor')
        ], function () {
            return Apartment::getObjectFloors();
        }, function ($value) { // if the filter is active
            $this->crud->addClause('where', 'floor', $value);
        });

        $this->crud->addFilter([ // select2 filter
            'name' => 'number',
            'type' => 'select2',
            'label' => __('apartments.crud_fields.number')
        ], function () {
            return Apartment::getObjectFlats();
        }, function ($value) { // if the filter is active
            $this->crud->addClause('where', 'number', $value);
        });

         // ------ CRUD CUSTOM VIEWS
        $this->crud->setListView('crud::apartments.list');
        $this->crud->setShowView('crud::apartments.show');
        $this->data['pageLengthMenu'] = $this->getPageLengthMenu();
        $this->crud->setDefaultPageLength(self::DEFAULT_PAGE_LENGTH);

        $this->crud->setCreateView('crud::create');
    }

    public function edit($id)
    {
        if(preg_match('/[a-zA-Zа-яА-Я]+/i',$id)){
            Abort(404);
        }
        $min_activation_limit = $this->crud->getModel()::findOrFail($id)->getActivationsUsed();

        $this->crud->addField([
            'name' => 'activation_limit',
            'label' => __('apartments.crud_fields.activation_limit'),
            'type' => 'number',
        ])->afterField('activation_code');

        // get the info for that entry
        $this->data['entry'] = $this->crud->getEntry($id);

        /*        $this->crud->addField([
                    'name' => 'cameras',
                    'type' => 'hidden-textarea',
                    'attributes' => [
                        'id' => 'camera-field'
                    ],
                    'value' => json_encode($this->data['entry']->cameras()->get())
                ]);

                $this->crud->addField([
                    'name' => 'calling_panels',
                    'type' => 'hidden-textarea',
                    'attributes' => [
                        'id' => 'callingPanel-field'
                    ],
                    'value' => json_encode($this->data['entry']->calling_panels()->get())
                ]);*/


        if ($this->data['entry']->object_id != config('app.object.id')) {
            return \Redirect::to($this->crud->route);
        }
        $this->crud->hasAccessOrFail('update');
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['title'] = trans('backpack::crud.edit') . ' ' . $this->crud->entity_name;

        $this->data['id'] = $id;

        $this->data['sip_numbers'] = ApartmentSipNumber::where('apartment_id', $id)->get();

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getEditView(), $this->data);
    }

    public function store(StoreRequest $request)
    {
        $helper = new AcsHttpRequestHelper();
        DB::beginTransaction();
        $redirect_location = $this->storeCrud($request);
        // your additional operations after save here
        // use $this->data['entry'] or $this->crud->entry
        $response = $helper->sendPost('/api/apartment', [[
            'object_id' => config('app.object.id'),
            'ds_id' => $this->crud->entry->id,
            'entrance' => $this->crud->entry->entrance,
            'floor' => $this->crud->entry->floor,
            'number' => $this->crud->entry->number,
            'available_activations_count' => $this->crud->entry->activation_limit
        ]]);
        switch ($response['status']) {
            case 200:
            case 201:
                DB::commit();
                // show a success message
                \Alert::success(trans('backpack::crud.insert_success'))->flash();
                // your additional operations before save here
                return $redirect_location;
                break;
            case 471:
                DB::rollBack();
                return back()->withInput()->withErrors([
                    'activation_limit' => __('apartments.validation.too_much_activation')
                ]);
                break;
            default:
                DB::rollBack();
                \Alert::error(__('response_error.' . $response['status']))->flash();
                return back()->withInput();
                break;
        }
    }

    public function create()
    {
        $this->crud->hasAccessOrFail('create');

        $this->crud->addField([
            'name' => 'activation_limit',
            'label' => __('apartments.crud_fields.activation_limit'),
            'type' => 'number',
            'value' => 3,
            'attributes' => [
                'min' => 0
            ]
        ])->afterField('number');

        // prepare the fields you need to show
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getCreateFields();
        $this->data['title'] = trans('backpack::crud.add') . ' ' . $this->crud->entity_name;

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getCreateView(), $this->data);
    }

    public function update(UpdateRequest $request)
    {
        DB::beginTransaction();
        $issetSipNumbers = (new ApartmentSipNumber())->where('apartment_id', $request->input('id'))->get()->toArray();
        // your additional operations before save here

        // Исправление отсутствия изменения номера квартиры на странице уведомлений и отзывов
        $number = Apartment::find($request->id)->number;

        $notices = Notice::where('apartment', $number)
            ->where('object_id', config('app.object.id'))
            ->get();
        $claims = Review::where('apartment', $number)
            ->where('object_id', config('app.object.id'))
            ->get();

        $notices->transform(function($notice) use ($request) {
            $notice->apartment = $request->get('number');
            return $notice;
        });

        $claims->transform(function($claim) use ($request) {
            $claim->apartment = $request->get('number');
            return $claim;
        });

        $notices->map(function ($notice) {
            $notice->save();
        });

        $claims->map(function ($claim) {
            $claim->save();
        });

        $redirect_location = $this->updateCrud($request);

        $response = (new AcsHttpRequestHelper())->sendPut('/api/apartment/' . $this->data['entry']->id, [
            'entrance' => $this->crud->entry->entrance,
            'floor' => $this->crud->entry->floor,
            'number' => $this->crud->entry->number,
            'available_activations_count' => $this->crud->entry->activation_limit
        ]);
        switch ($response['status']) {
            case 200:
                DB::commit();
                \Alert::success(trans('backpack::crud.insert_success'))->flash();
                $this->updateOrCreateSipNumbers(json_decode($request->input('sip_numbers')), $issetSipNumbers);
                $this->attachSipNumbers($this->crud->entry->id);
                return $redirect_location;
                break;
            case 471:
                DB::rollBack();
                return back()->withInput()->withErrors([
                    'activation_limit' => __('apartments.validation.too_much_activation')
                ]);
                break;
            case 472:
                DB::rollBack();
                return back()->withInput()->withErrors([
                    'activation_limit' => __('apartments.validation.activation_limit')
                ]);
                break;
            default:
                DB::rollBack();
                \Alert::error(__('response_error.' . $response['status']))->flash();
                return back()->withInput();
                break;
        }

    }

    private function attachSipNumbers($apartmentId)
    {
        $apartment = Apartment::find($apartmentId);
        $sipNumber = new ApartmentSipNumber;
        $busySipNumbers = $this->getBusySipNumbers($apartment);
        $allApartmentSipNumbers = ApartmentSipNumber::where('apartment_id', $apartmentId)->pluck('id')->toArray();
        $freeSipNumbers = [];
        foreach ($allApartmentSipNumbers as $number) {
            if (!in_array($number, $busySipNumbers)) {
                $freeSipNumbers[] = $number;
            }
        }
        $users = $apartment->users()->withCount('tokens')->get();
        foreach ($users as $user) {
            $devices = $user->devices()->get();
            foreach ($devices as $device) {
                if ($device->sip_number != null) {
                    continue;
                } else {
                    $sip_number = array_pop($freeSipNumbers);
                    if ($sip_number) {
                        $sip = $sipNumber->find($sip_number);
                        $device->sip_number_id = $sip->id;
                        $device->save();
                    }
                }

            }
        }
    }

    private function getBusySipNumbers(Apartment $apartment)
    {
        $busySipNumbers = [];
        $users = $apartment->users()->withCount('tokens')->get();
        foreach ($users as $user) {
            $numbers = $user->devices()->whereNotNull('sip_number_id')->pluck('sip_number_id')->toArray();
            $busySipNumbers = array_merge($busySipNumbers, $numbers);
        }
        return $busySipNumbers;
    }

    private function updateOrCreateSipNumbers($newSipNumbers, $issetSipNumbers = [])
    {
        $newSipNumbers = $newSipNumbers ?: [];

        $snIds = [];
        $sip_number = new ApartmentSipNumber;
        foreach ($newSipNumbers as $number) {
            if (isset($number->sip_number)) {
                $edited_sip_number = ApartmentSipNumber::getSipNumberWithObjectId($number->sip_number);

                if (!$edited_sip_number) {
                    $sip_number->create(['apartment_id' => $this->crud->entry->id, 'sip_number' => $number->sip_number, 'sip_password' => $number->sip_password]);
                    $snIds[] = $sip_number->sip_number;
                } else {
                    $edited_sip_number->sip_password = $number->sip_password;
                    $edited_sip_number->save();
                    $snIds[] = $edited_sip_number->sip_number;
                }
            }
        }
        if ($issetSipNumbers) {
            foreach ($issetSipNumbers as $issetSipNumber) {

                if (!in_array($issetSipNumber['sip_number'], $snIds)) {
                    $sip_number->find($issetSipNumber['id'])->delete();
                }
            }
        }
    }

    /**
     * @param int[] $lengthMenu
     */
    public function setPageLengthMenu(array $lengthMenu)
    {
        $this->lengthMenu = $lengthMenu;
    }

    /**
     * @return int[]
     */
    public function getPageLengthMenu()
    {
        return $this->lengthMenu;
    }

    public function resetActivations($id)
    {

        $apartment = $this->crud->getModel()->findOrFail($id);
        $response = (new AcsHttpRequestHelper())->sendDelete('/api/apartment/' . $apartment->id . '/activations', []);
        switch ($response['status']) {
            case 200:
                foreach ($apartment->users as $user) {
                    $user->tokens()->delete();
                }
                return ['activations_used' => $apartment->getActivationsUsed()];
                break;
            default:
                return response()->json([],$response['status']);
                break;
        }
    }

    public function resetActivationCode($id)
    {
        $apartment = $this->crud->getModel()->findOrFail($id);
        $response = (new AcsHttpRequestHelper())->sendPost('/api/apartment/' . $apartment->id . '/refresh', []);
        switch ($response['status']) {
            case 200:
                $apartment->activation_code = null;
                $apartment->save();
                foreach ($apartment->users as $user) {
                    $user->tokens()->delete();
                }
                return back();
                break;
            default:
                \Alert::error(__('response_error.' . $response['status']))->flash();
                return back()->withInput();
                break;
        }
    }

    /**
     * Get the save configured save action or the one stored in a session variable.
     * @return [type] [description]
     */
    public function getSaveAction()
    {
        $saveAction = session('save_action', config('backpack.crud.default_save_action', 'save_and_back'));
        $saveOptions = [];
        $saveCurrent = [
            'value' => $saveAction,
            'label' => $this->getSaveActionButtonName($saveAction),
        ];

        switch ($saveAction) {
            case 'save_and_edit':
                $saveOptions['save_and_back'] = $this->getSaveActionButtonName('save_and_back');
                if (Auth::user()->can(Permission::APARTMENTS_CREATE_ACCESS)) {
                    $saveOptions['save_and_new'] = $this->getSaveActionButtonName('save_and_new');
                }
                break;
            case 'save_and_new':
                if (Auth::user()->can(Permission::APARTMENTS_CREATE_ACCESS)) {
                    $saveOptions['save_and_back'] = $this->getSaveActionButtonName('save_and_back');
                }
                if (Auth::user()->can(Permission::APARTMENTS_EDIT_ACCESS)) {
                    $saveOptions['save_and_edit'] = $this->getSaveActionButtonName('save_and_edit');
                }
                break;
            case 'save_and_back':
            default:
                if (Auth::user()->can(Permission::APARTMENTS_EDIT_ACCESS)) {
                    $saveOptions['save_and_edit'] = $this->getSaveActionButtonName('save_and_edit');
                }
                if (Auth::user()->can(Permission::APARTMENTS_CREATE_ACCESS)) {
                    $saveOptions['save_and_new'] = $this->getSaveActionButtonName('save_and_new');
                }
                break;
        }

        return [
            'active' => $saveCurrent,
            'options' => $saveOptions,
        ];
    }

    /**
     * Get the translated text for the Save button.
     * @param  string $actionValue [description]
     * @return [type]              [description]
     */
    private function getSaveActionButtonName($actionValue = 'save_and_back')
    {
        switch ($actionValue) {
            case 'save_and_edit':
                return trans('backpack::crud.save_action_save_and_edit');
                break;
            case 'save_and_new':
                return trans('backpack::crud.save_action_save_and_new');
                break;
            case 'save_and_back':
            default:
                return trans('backpack::crud.save_action_save_and_back');
                break;
        }
    }

    public function storeCrud(StoreCrudRequest $request = null)
    {
        $this->crud->hasAccessOrFail('create');

        // fallback to global request instance
        if (is_null($request)) {
            $request = \Request::instance();
        }

        // replace empty values with NULL, so that it will work with MySQL strict mode on
        foreach ($request->input() as $key => $value) {
            if (empty($value) && $value !== '0') {
                $request->request->set($key, null);
            }
        }

        // insert item in the db
        $item = $this->crud->create($request->except(['save_action', '_token', '_method', 'current_tab']));
        $this->data['entry'] = $this->crud->entry = $item;

        $this->updateOrCreateSipNumbers(json_decode($request->input('sip_numbers')));
        $this->attachSipNumbers($this->crud->entry->id);

        // save the redirect choice for next time
        $this->setSaveAction();

        return $this->performSaveAction($item->getKey());
    }

    /**
     * Update the specified resource in the database.
     *
     * @param UpdateRequest $request - type injection used for validation using Requests
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateCrud(UpdateCrudRequest $request = null)
    {
        $this->crud->hasAccessOrFail('update');

        // fallback to global request instance
        if (is_null($request)) {
            $request = \Request::instance();
        }

        // replace empty values with NULL, so that it will work with MySQL strict mode on
        foreach ($request->input() as $key => $value) {
            if (empty($value) && $value !== '0') {
                $request->request->set($key, null);
            }
        }

        // update the row in the db
        $item = $this->crud->update($request->get($this->crud->model->getKeyName()),
            $request->except('save_action', '_token', '_method', 'current_tab'));
        $this->data['entry'] = $this->crud->entry = $item;

        // save the redirect choice for next time
        $this->setSaveAction();

        return $this->performSaveAction($item->getKey());
    }

    public function index()
    {
        $this->crud->hasAccessOrFail('list');

        $this->data['crud'] = $this->crud;
        $this->crud->addButton('top', 'activity', 'view', 'crud::buttons.activity', 'end');
        $this->data['title'] = ucfirst($this->crud->entity_name_plural);
        $response = (new AcsHttpRequestHelper(true, ['guzzle' => false]))->sendGet('/api/activations');
        switch ($response['status']) {
            case 200:
                $body = !is_null($response['body']);
                $this->data['acs_server_answer'] = $body;

                if ($body) {
                    $this->data['used_activations_count'] = $response['body']->used_activations_count;
                    $this->data['available_activations_count'] = $response['body']->available_activations_count;
                    $this->data['used_by_users_count'] =
                        isset($response['body']->used_by_users_count) ?
                            $response['body']->used_by_users_count : 0;
                } else {
                    $this->data['used_activations_count'] = $this->data['available_activations_count'] = $this->data['used_by_users_count'] = null;
                    \Alert::error(__('response_error.0'))->flash();
                }
                // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
                return view($this->crud->getListView(), $this->data);
                break;

            case 421:
                return view($this->crud->getListView(), $this->data);
                break;
            default:
                \Alert::error(__('response_error.' . $response['status']))->flash();
                return view($this->crud->getListView(), $this->data);
                break;
        }
    }

    public function show($id)
    {
        $this->crud->hasAccessOrFail('show');

        // get entry ID from Request (makes sure its the last ID for nested resources)
        $id = $this->crud->getCurrentEntryId() ?? $id;

        // set columns from db
        $this->crud->setFromDb();
        $this->crud->removeColumn('object_id');
        $this->crud->removeColumn('activation_limit');

        $this->crud->addColumn([
            'name' => 'sip_numbers',
            'label' => __('apartments.crud_fields.sip_numbers'),
            'type' => 'sip_numbers',
            'function_name' => 'getSipNumbersHtml',
        ]);

        // cycle through columns
        foreach ($this->crud->columns as $key => $column) {
            // remove any autoset relationship columns
            if (array_key_exists('model', $column) && array_key_exists('autoset', $column) && $column['autoset']) {
                $this->crud->removeColumn($column['name']);
            }

            // remove the row_number column, since it doesn't make sense in this context
            if ($column['type'] == 'row_number') {
                $this->crud->removeColumn($column['name']);
            }
        }

        // get the info for that entry
        $this->data['entry'] = $this->crud->getEntry($id);
        $this->data['crud'] = $this->crud;
        $this->data['title'] = trans('backpack::crud.preview').' '.$this->crud->entity_name;

        // remove preview button from stack:line
        $this->crud->removeButton('preview');
        $this->crud->removeButton('delete');

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getShowView(), $this->data);
    }

    /**
     * The search function that is called by the data table.
     *
     * @return  JSON Array of cells in HTML form.
     */
    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // overwrite any order set in the setup() method with the datatables order
        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);
            if ($column['tableColumn']) {
                // clear any past orderBy rules
                $this->crud->query->getQuery()->orders = null;
                // apply the current orderBy rules
                if(in_array($column['name'], ['entrance', 'floor', 'number'])) {
                    // для поля с именем number установлена отдельная сортировка для более удобочитаемого отображения квартир, имеющих номера
                    $this->crud->query = $this->crud->query->orderByRaw(
                        $column['name'] . '+0 '.$column_direction.', '.
                        $column['name'].' '.$column_direction.
                        ', LENGTH('.$column['name'].') '.$column_direction
                    );
                } else {
                    $this->crud->orderBy($column['name'], $column_direction);
                }
            }
        }
        $entries = $this->crud->getEntries();

        return $this->crud->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);
    }
}
