<?php

namespace App\Http\Controllers\Admin;

use App\Models\Advert;
use App\Models\AdvertDestination;
use App\Models\AdvertViewed;
use App\Models\Apartment;
use App\Models\Category;
use App\Models\MessageTemplate;
use App\Models\PermissionManager\Permission;
use App\Models\User;
use App\Traits\Admin\Ordering;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\CRUD\app\Http\Controllers\CrudController;

// VALIDATION: change the requests to match your own file names if you need form validation
use App\Http\Requests\AdvertRequest as StoreRequest;
use App\Http\Requests\AdvertRequest as UpdateRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;


class AdvertCrudController extends CrudController {

    use CheckingPermissions, Ordering;

    /**
     * AdvertCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::ANNOUNCEMENT_FULL_ACCESS],
            [
                'create',
                'store',
                'index',
                'edit',
                'update',
                'destroy',
                'printAdvert',
                'hideAdvert',
                'results',
            ],
        ]]);
    }

    /**
     * @var string[] $displayFields
     */
    public static $displayFields = [
        'theme',
        'text',
        'created_at',
        'actual_from_date',
        'actual_to_date',
    ];

    public function setup() {

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\Advert');
        $this->crud->setRoute(config('backpack.base.route_prefix') . '/announcement');
        $this->crud->setEntityNameStrings(__('adverts.adverts_singular'), __('adverts.adverts_plural'));

        $this->applyBackpackPermissions();

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */

        // ------ CRUD FIELDS

        $this->crud->addField([
            'name' => 'object_id',
            'type' => 'hidden',
            'value' => config('app.object.id')
        ]);

        $this->crud->addField([
            'name' => 'destinations',
            'label' => __('adverts.form_fields.destination_edit'),
            'type' => 'hidden',
            'attributes' => [
                'id' => 'destinations-field',
            ],
        ]);

        $apartment = new Apartment;

        $this->crud->addField([
            'name' => 'destinations-list',
            'type' => 'custom_html',
            'is_destination' => TRUE,
            'value' => '<label>' . __('adverts.form_fields.destination') . '</label><div class="destinations-list"></div>',
        ]);

        $this->crud->addField([
            'name' => 'is_all_house',
            'label' => __('adverts.form_fields.is_all_house'),
            'type' => 'checkbox',
            'col' => 'col1',
            'is_destination' => TRUE,
            'attributes' => [
                'id' => 'is_all_house-field',
            ],
        ], 'create');

        $this->crud->addField([
            'name' => 'entrance',
            'label' => __('adverts.form_fields.select_entrance'),
            'type' => 'select2_from_array',
            'options' => $apartment->getEntrances(),
            'allows_null' => TRUE,
            'col' => 'col1',
            'is_destination' => TRUE,
            'attributes' => [
                'id' => 'entrance-field',
                'data-disable' => 'is_all_house',
            ],
        ]);

        $this->crud->addField([
            'name' => 'floor',
            'label' => __('adverts.form_fields.select_floor'),
            'type' => 'select2_from_array',
            'options' => $apartment->getFloors(),
            'allows_null' => TRUE,
            'col' => 'col1',
            'is_destination' => TRUE,
            'attributes' => [
                'id' => 'floor-field',
                'data-disable' => 'is_all_house',
            ],
        ]);

        $this->crud->addField([
            'name' => 'add-destination-group',
            'col' => 'col1',
            'is_destination' => TRUE,
            'type' => 'custom_html',
            'value' => '<button id="add-destination-group" class="btn btn-primary" data-disable="is_all_house">' . __('destinations.add_destination_group') . '</button>',
        ], 'create');

        $this->crud->addField([
            'name' => 'apartment',
            'label' => __('adverts.form_fields.select_apartment'),
            'type' => 'select2_from_array',
            'col' => 'col2',
            'is_destination' => TRUE,
            'options' => $apartment->getFlats(),
            'allows_null' => TRUE,
            'attributes' => [
                'id' => 'apartment-field',
                'data-disable' => 'is_all_house',
            ],
        ]);

        $this->crud->addField([
            'name' => 'add-destination',
            'type' => 'custom_html',
            'col' => 'col2',
            'is_destination' => TRUE,
            'value' => '<button id="add-destination" class="btn btn-primary" data-disable="is_all_house">' . __('destinations.add_destination') . '</button>',
        ], 'create');


        $this->crud->addField([
            'name' => 'advert_date_range',
            'start_name' => 'actual_from_date',
            'end_name' => 'actual_to_date',
            'label' => __('adverts.crud_fields.advert_date_range'),
            'type' => 'date_range',
            'start_default' => date('Y-m-d'),
            'end_default' => date('Y-m-d', strtotime('+1 day')),
            'date_range_options' => [
                'locale' => ['format' => 'DD.MM.YYYY'],
            ],
        ], 'create');

        $this->crud->addField([
            'name' => 'actual_from_date',
            'label' => __('adverts.crud_fields.actual_from_date'),
            'type' => 'text',
            'attributes' => [
                'id' => 'actual_from_date-field',
                'readonly' => TRUE,
            ],
        ], 'update');

        $this->crud->addField([
            'name' => 'actual_to_date',
            'label' => __('adverts.crud_fields.actual_to_date'),
            'type' => 'text',
            'attributes' => [
                'id' => 'actual_to_date-field',
                'readonly' => TRUE,
            ],
        ], 'update');

        $this->crud->addField([
            'name' => 'template',
            'value' => '<button class="btn btn-primary" type="button" data-toggle="modal" data-target="#templates-picker">'
                . __('adverts.buttons.template') . '</button>',
            'type' => 'custom_html',
        ]);

        $this->crud->addField([
            'name' => 'theme',
            'label' => __('adverts.crud_fields.theme'),
            'type' => 'text',
            'attributes' => [
                'id' => 'theme-field',
            ],
        ]);

        $this->crud->addField([
            'name' => 'text',
            'label' => __('adverts.crud_fields.text'),
            'type' => 'textarea',
            'attributes' => [
                'id' => 'text-field',
            ],
        ]);

        // ------ CRUD COLUMNS

        $this->crud->addClause('where', 'object_id', '=', config('app.object.id'));

        $this->crud->addColumn([
            'name' => 'whom',
            'label' => __('adverts.crud_fields.whom'),
            'type' => 'model_function',
            'function_name' => 'getWhom'
        ]);

        foreach (self::$displayFields as $field) {
            $localizedFieldName = __("adverts.crud_fields.$field");
            if ($field == 'actual_to_date') {
                $this->crud->addColumn([
                    'name' => $field,
                    'label' => $localizedFieldName,
                    'type' => 'model_function',
                    'function_name' => 'getActualTimeTo',
                ]);
            } elseif ($field == 'created_at') {
                $this->crud->addColumn([
                    'name' => $field,
                    'label' => $localizedFieldName,
                    'type' => 'text_datetime_utc',
                ]);
            } elseif ($field == 'actual_to_date') {
                $this->crud->addColumn([
                    'name' => $field,
                    'label' => $localizedFieldName,
                    'type' => 'advert_actual_to_date',
                ]);
            } else {
                $this->crud->addColumn([
                    'name' => $field,
                    'label' => $localizedFieldName,
                    'type' => 'text',
                ]);
            }
        }

        // ------ CRUD BUTTONS

        $this->crud->removeButton('delete');
        $this->crud->removeButton('update');
        $this->crud->removeButton('preview');
        $this->crud->addButton('line', 'show', 'view', 'vendor.backpack.crud.adverts.buttons.preview', 'top');
        $this->crud->addButton('line', 'results', 'view', 'vendor.backpack.crud.adverts.buttons.results'); // add a button; possible types are: view, model_function
        $this->crud->addButton('line', 'hide', 'view', 'vendor.backpack.crud.adverts.buttons.delete', 'end'); // add a button; possible types are: view, model_function

        // possible positions: 'beginning' and 'end'; defaults to 'beginning' for the 'line' stack, 'end' for the others;
        // $this->crud->addButton($stack, $name, $type, $content, $position); // add a button; possible types are: view, model_function
        // $this->crud->addButtonFromModelFunction($stack, $name, $model_function_name, $position); // add a button whose HTML is returned by a method in the CRUD model
        // $this->crud->addButtonFromView($stack, $name, $view, $position); // add a button whose HTML is in a view placed at resources\views\vendor\backpack\crud\buttons
        // $this->crud->removeButton($name);
        // $this->crud->removeButtonFromStack($name, $stack);
        // $this->crud->removeAllButtons();
        // $this->crud->removeAllButtonsFromStack('line');

        // ------ CRUD ACCESS
        $this->crud->allowAccess(['show','hide']);
        // $this->crud->allowAccess(['list', 'create', 'update', 'reorder', 'delete']);
        // $this->crud->denyAccess(['list', 'create', 'update', 'reorder', 'delete']);

        // ------ CRUD REORDER
        // $this->crud->enableReorder('label_name', MAX_TREE_LEVEL);
        // NOTE: you also need to do allow access to the right users: $this->crud->allowAccess('reorder');

        // ------ CRUD DETAILS ROW
        // $this->crud->enableDetailsRow();
        // NOTE: you also need to do allow access to the right users: $this->crud->allowAccess('details_row');
        // NOTE: you also need to do overwrite the showDetailsRow($id) method in your EntityCrudController to show whatever you'd like in the details row OR overwrite the views/backpack/crud/details_row.blade.php

        // ------ REVISIONS
        // You also need to use \Venturecraft\Revisionable\RevisionableTrait;
        // Please check out: https://laravel-backpack.readme.io/docs/crud#revisions
        // $this->crud->allowAccess('revisions');

        // ------ AJAX TABLE VIEW
        // Please note the drawbacks of this though:
        // - 1-n and n-n columns are not searchable
        // - date and datetime columns won't be sortable anymore
        // $this->crud->enableAjaxTable();

        // ------ DATATABLE EXPORT BUTTONS
        // Show export to PDF, CSV, XLS and Print buttons on the table view.
        // Does not work well with AJAX datatables.
        // $this->crud->enableExportButtons();

        // ------ ADVANCED QUERIES
        // $this->crud->addClause('active');
        // $this->crud->addClause('type', 'car');
        // $this->crud->addClause('where', 'name', '==', 'car');
        // $this->crud->addClause('whereName', 'car');
        // $this->crud->addClause('whereHas', 'posts', function($query) {
        //     $query->activePosts();
        // });
        // $this->crud->addClause('withoutGlobalScopes');
        // $this->crud->addClause('withoutGlobalScope', VisibleScope::class);
        // $this->crud->with(); // eager load relationships
        // $this->crud->orderBy();
        // $this->crud->groupBy();
        // $this->crud->limit();
        $this->crud->setListView('crud::adverts.list');
        $this->crud->orderBy('created_at', 'DESC');
        // ------ CRUD CUSTOM VIEWS
        $this->crud->setCreateView('crud::adverts.create');
        $this->crud->setEditView('crud::adverts.edit');
        $this->crud->setShowView('crud::adverts.show');
    }

    public function store(StoreRequest $request) {
        $saveAction = \Request::input('save_action');

        $isAllHouse = $request->get('is_all_house') == 1;
        if ($isAllHouse) {
            $request->merge(['destinations' => \GuzzleHttp\json_encode([['is_all_house' => TRUE]])]);
        }

        $redirect = parent::storeCrud($request);

        $itemId = $this->data['entry']->id;
        if ($saveAction === 'save_and_print') {
            $redirect = \Redirect::to($this->crud->route . '/' . $itemId . '/print');
        }

        $destinations = \GuzzleHttp\json_decode($request->get('destinations'), TRUE);

        foreach ($destinations as $destination) {
            $destination['advert_id'] = $itemId;
            (new AdvertDestination)->create($destination);
        }

        return $redirect;
    }

    public function update(UpdateRequest $request) {
        $saveAction = \Request::input('save_action');
        $itemId = \Request::input('id');

        $redirect = parent::updateCrud($request);
        if ($saveAction === 'save_and_print') {
            $redirect = \Redirect::to($this->crud->route . '/' . $itemId . '/print');
        }

        return \Redirect::to($this->crud->route);
    }

    public function printAdvert(Advert $advert) {
        return view('crud::adverts.print', [
            'advert' => $advert,
            'companyName' => config('company.name'),
            'backUrl' => '/' . $this->crud->route . '/' . $advert->id,
        ]);
    }

    public function hideAdvert(Advert $advert) {
        $time = date('Y-m-d', time() - 86400);
        $advert->actual_to_date = $time;

        if ($advert->save()) {
            $message = trans('adverts.action_successful_hide');

            return [
                'result' => $message,
                'time' => $time
            ];
        }
        return response()->json([], 503);
    }

    /**
     * Show the form for creating inserting a new row.
     *
     * @return Response
     */
    public function create() {
        $this->crud->hasAccessOrFail('create');

        // prepare the fields you need to show
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getCreateFields();
        $this->data['title'] = trans('backpack::crud.add') . ' ' . $this->crud->entity_name;

        $this->setTemplatesData();

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getCreateView(), $this->data);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     *
     * @return Response
     */
    public function edit($id) {
        $this->crud->hasAccessOrFail('update');

        // get the info for that entry
        $this->data['entry'] = $this->crud->getEntry($id);
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['title'] = trans('backpack::crud.edit') . ' ' . $this->crud->entity_name;

        $this->data['id'] = $id;

        $this->setTemplatesData();

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getEditView(), $this->data);
    }

    public function show($id) {
        $this->crud->hasAccessOrFail('show');
        $this->data['entry'] = $this->crud->getEntry($id);
        $this->data['crud'] = $this->crud;
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['title'] = trans('backpack::crud.preview') . ' ' . $this->crud->entity_name;
        unset($this->data['fields']['template']);
        foreach ($this->data['fields'] as $key => &$field) {
            $field['attributes']['readonly'] = true;
        }
        $this->data['id'] = $id;
        return view($this->crud->getShowView(), $this->data);
    }


    public function results($id) {
        $this->prepareResultsData($id);
        return view('crud::adverts.results',
            $this->data
        );
    }

    /**
     * Preparing data for displaying on the results page
     *
     * @param $id
     * @return \Illuminate\Support\Collection
     */
    private function prepareResultsData($id)
    {
        $this->data['title'] = __('adverts.crud.results');

        // First you need to get a list of current apartments, floors and entrances
        $destinations = AdvertDestination::where('advert_id', $id)->get();
        $apartments = Apartment::where('object_id', config('app.object.id'));

        // In all house
        $in_all_house = boolval($destinations->pluck('is_all_house')->filter()->first());

        // If not sent to the whole house
        if (!$in_all_house) {

            // Go through the array of destinations and make a request for all the recipients
            // In builder it is not possible to collect a query of the form where ... and ((... and ...) or (...))
            // therefore, we build a query using the sql language
            $destinations_sql = null;
            foreach ($destinations as $key => $destination) {
                // According to the current logic, if the apartment is set, then the fields with the floor and the entrance are empty
                // After replacing apartment, number with apartment_id, you need to make changes to this code
                if (!is_null($destination->apartment)) {

                    $destinations_sql .= ($key !== 0 ? ' OR ' : '') . '(number = ?)';
                    $apartments->addBinding($destination->apartment);

                } elseif (!is_null($destination->entrance)
                    && !is_null($destination->floor)) {

                    $destinations_sql .= ($key !== 0 ? ' OR ' : '') . '(entrance = ? AND floor = ?)';
                    $apartments->addBinding($destination->entrance);
                    $apartments->addBinding($destination->floor);

                } elseif (!is_null($destination->entrance)
                    || !is_null($destination->floor)) {

                    if (!is_null($destination->entrance)) {
                        $destinations_sql .= ($key !== 0 ? ' OR ' : '') . '(entrance = ?)';
                        $apartments->addBinding($destination->entrance);
                    }

                    if (!is_null($destination->floor)) {
                        $destinations_sql .= ($key !== 0 ? ' OR ' : '') . '(floor = ?)';
                        $apartments->addBinding($destination->floor);
                    }

                } else {
                    $destinations_sql .= null;
                }
            }

            // If $destinations_sql is empty, then an additional condition must be excluded from the resulting request
            if ($destinations_sql) {
                $apartments
                    ->whereRaw('(' . $destinations_sql . ')');
            }
        }

        // Calculate the total number of ads posted
        $apartments_count_query = clone($apartments);
        $apartments_count = $apartments_count_query->count();

        // Exclude from the list of apartments that have already read ads
        $apartments
            ->whereNotIn('number', collect(DB::select('
                SELECT number FROM 
                    (
                        SELECT apartment_id 
                        FROM adverts_viewed 
                        LEFT JOIN apartment_user 
                            ON adverts_viewed.user_id = apartment_user.user_id 
                            WHERE adverts_viewed.advert_id = ?
                    ) AS announcement
                LEFT JOIN apartments
                  ON announcement.apartment_id = apartments.id
        ', [$id]))->pluck('number'));

        // Calculate the number of remaining apartments and apartments that have read the announcement
        $apartments_remaining_query = clone($apartments);
        $apartments_remaining = $apartments_remaining_query->count();

        $this->data['count_destination'] = $apartments_count;
        $this->data['count_viewed'] = $apartments_count - $apartments_remaining;
        $this->data['crud'] = $this->crud;

        // SQL statement for expression testing
        // dd($apartments->toSql(), $apartments->getBindings(), $apartments->get());

        return $this->data['apartments'] = $apartments->get();
    }

    protected function setTemplatesData() {
        $this->data['templatePickerTitle'] = trans('adverts.add_template');
        $this->data['templateStyle'] = Auth::user()->getStyleType();

        switch ($this->data['templateStyle']) {
            case 'list':
                $this->data['templates'] = MessageTemplate::all();

                break;
            case 'tags':

                $templateTags = MessageTemplate::where('tags', '!=', NULL)
                    ->get()
                    ->pluck('tags')
                    ->toArray();
                $tags = [];
                foreach ($templateTags as $tplTag) {
                    $tags = array_merge(explode(',', $tplTag), $tags);
                }
                $this->data['tags'] = array_values(array_unique($tags));

                $this->data['templates'] = MessageTemplate::all();

                break;

            case 'tree':
                $this->data['categories'] = collect(Category::all())
                    ->sortBy('lft')
                    ->keyBy((new Category)->getKeyName());
                foreach ($this->data['categories'] as $key => $entry) {
                    $entry->items = $entry->messageTemplates()->get();
                }
                break;
        }
    }


    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // Кастомная сортировка
        $sortedEntriesIds = null;
        $sortedEntries = collect();
        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);

            $customOrder = true;
            $table = $this->crud->getModel()->getTable();
            switch ($column['name']) {
                // По столбцу тема
                case 'theme':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'theme');
                    break;
                // По столбцу оюъявление
                case 'text':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'text');
                    break;
                default:
                    $customOrder = false;
                    if ($column['tableColumn']) {
                        // clear any past orderBy rules
                        $this->crud->query->getQuery()->orders = null;
                        // apply the current orderBy rules
                        $this->crud->orderBy($column['name'], $column_direction);
                    }
                    break;
            }

            // Если сортировка кастомная, то сформировать коллекцию объектов
            if ($customOrder) {
                $sortedEntries = $this->createOrderedObjectsByIds('App\Models\Advert', $sortedEntriesIds);
            }
        }

        // Если сортировка кастомная, то присваиваем отсортированные записи
        $entries = $sortedEntries->count()? $sortedEntries : $this->crud->getEntries();

        $data = $this->crud->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);



        /* Выделение жирным необходимых полей
         *
         *  Сформировать массив идентификаторов, которые будут выделяться жирным
         * isActive - которые будут выделяться жирным
         * isWillActive - которые будут выделяться очень жирным
         */
        $entries = $this->crud->getEntries();

        $isActive = collect();
        $isWillActive = collect();

        for ($i = 0; $i < count($entries); $i++) {
            if ($entries[$i]->isActive()) {
                $isActive->push($i);
            } elseif ($entries[$i]->isWillActive()) {
                $isWillActive->push($i);
            }
        }

        foreach ($isActive as $activeId) {
            $data['data'][$activeId]['DT_RowClass'] = 'font-weight-900';
        }

        foreach ($isWillActive as $willActiveId) {
            $data['data'][$willActiveId]['DT_RowClass'] = 'font-weight-600';
        }

        $i = 0;
        foreach ($entries as $entry) {
            $data['data'][$i]['DT_RowAttr'] = [
                'data-id' => $entry->id
            ];
            $i++;
        }

        return $data;
    }
}