<?php

namespace evApps\Installer\app\Http\Utilities;

use App\Helpers\AcsHttpRequestHelper;
use App\Models\Building as ObjectModel;
use App\Models\PermissionManager\Permission;
use App\Models\User;
use Illuminate\Encryption\Encrypter;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\URL;

/**
 * Class Installer
 *
 * @package evApps\Installer\app\Http\Utilities
 */
class Installer
{
    const REQUIREMENTS_STEP = 'requirements';
    const LICENSE_KEY_STEP = 'license-key';
    const DATABASE_STEP = 'database';
    const FIRST_OBJECT_STEP = 'first-object';
    const ADMIN_STEP = 'admin';
    const FINISH_STEP = 'finish';

    const STEPS = [
        self::REQUIREMENTS_STEP,
        self::LICENSE_KEY_STEP,
        self::DATABASE_STEP,
        self::FIRST_OBJECT_STEP,
        self::ADMIN_STEP,
        self::FINISH_STEP,
    ];


    const DB_CONNECTION_MYSQL = 'mysql';
    const DB_CONNECTION_SQLITE = 'sqlite';
    const DB_CONNECTION_PGSQL = 'pgsql';
    const DB_CONNECTION_SQLSRV = 'sqlsrv';

    const DB_CONNECTIONS = [
        self::DB_CONNECTION_MYSQL,
        self::DB_CONNECTION_SQLITE,
        self::DB_CONNECTION_PGSQL,
        self::DB_CONNECTION_SQLSRV,
    ];


    const SESSION_BASE_KEY = 'evapps.installer.installation';


    /**
     * @param string $step
     *
     * @return bool
     */
    public static function completeStep(string $step)
    {
        if (in_array($step, self::STEPS)) {
            Session::put(self::SESSION_BASE_KEY . '.' . $step . '.completed', true);

            return true;
        }

        return false;
    }

    /**
     * @param string $step
     * @param array $data
     *
     * @return bool
     */
    public static function saveStepData(string $step, array $data)
    {
        if (in_array($step, self::STEPS)) {
            Session::put(self::SESSION_BASE_KEY . '.' . $step . '.data', $data);

            return true;
        }

        return false;
    }

    /**
     * @param string $step
     *
     * @return bool
     */
    public static function isStepCompleted(string $step)
    {
        return (boolean)Session::get(self::SESSION_BASE_KEY . '.' . $step . '.completed');
    }

    /**
     * @param string $step
     *
     * @return mixed
     */
    public static function getStepData(string $step)
    {
        return Session::get(self::SESSION_BASE_KEY . '.' . $step . '.data');
    }

    /**
     * @return bool
     */
    public static function isAlreadyInstalled()
    {
        return !!env('APP_INSTALLED');
    }

    /**
     * @return bool
     */
    public static function isFirstObjectCreated()
    {
        return !!env('APP_FIRST_OBJECT_CREATED', false);
    }

    public static function checkEnvFile()
    {
        if (empty(env('APP_KEY'))) {
            Artisan::call('key:generate', [
                '--force' => true,
            ]);

            static::updateEnv([
                'APP_URL' => URL::to('/'),
            ]);

            Artisan::call('config:clear');

            return Redirect::to('/install/clearCookies')->send();
        }
    }

    public static function createDefaultEnvFileIfNotExists()
    {
        if (!File::exists(base_path('.env'))) {
            $environmentEnvStubFile = base_path('.env.' . App::environment());
            if (File::exists($environmentEnvStubFile)) {
                File::copy($environmentEnvStubFile, base_path('.env'));
            } else {
                File::copy(base_path('.env.example'), base_path('.env'));
            }

            $encryptionKey = self::generateRandomEncryptionKey(Config::get('app.cipher'));
            Config::set('app.key', $encryptionKey);

            // Update .env file
            static::updateEnv([
                'APP_KEY' => $encryptionKey,
                'APP_URL' => url('/'),
            ]);
        }
    }

    /**
     * Generate a random encryption key for the application.
     *
     * @param string $cipher
     *
     * @return string
     */
    public static function generateRandomEncryptionKey(string $cipher)
    {
        return 'base64:' . base64_encode(Encrypter::generateKey($cipher));
    }

    public static function updateEnv($data)
    {
        if (empty($data) || !is_array($data) || !is_file(base_path('.env'))) {
            return false;
        }

        $env = file_get_contents(base_path('.env'));

        $env = explode("\n", $env);

        foreach ($data as $data_key => $data_value) {
            foreach ($env as $env_key => $env_value) {
                $entry = explode('=', $env_value, 2);

                // Check if new or old key
                if ($entry[0] == $data_key) {
                    $env[$env_key] = $data_key . '=' . $data_value;
                } else {
                    $env[$env_key] = $env_value;
                }
            }
        }

        $env = implode("\n", $env);

        file_put_contents(base_path('.env'), $env);

        return true;
    }

    /**
     * @return array
     */
    public static function checkServerRequirements()
    {
        $checkingResult = [
            'requirements' => ServerRequirementChecker::checkAll(),
            'checking_result' => true,
        ];

        foreach ($checkingResult['requirements'] as $requirement => $result) {
            if (!$result && ($requirement !== 'min_upload_file_size')) {
                $checkingResult['checking_result'] = false;
                break;
            }
        }

        return $checkingResult;
    }

    /**
     * @param string $licenseKey
     *
     * @return bool
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public static function checkLicenseKey(string $licenseKey)
    {
        $acsHttpRequestHelper = new AcsHttpRequestHelper(false);
        $response = $acsHttpRequestHelper->sendPost('/api/install', [
            'installer' => true,
            'licenseKeyHash' => md5($licenseKey),
        ]);

        if (isset($response['status'])) {
            if ($response['status'] === 200) {
              return true;
            } else if ($response['status'] === 409) {
                throw new \Exception(__(
                    'installer::license_key.license_key_already_used'
                ));
            } else if ($response['status'] === 421) {
                throw new \Exception(__(
                    'installer::license_key.server_is_not_active'
                ));
            }
        }

        return false;
    }

    /**
     * @param string $dbConnection
     * @param string $dbHost
     * @param string $dbPort
     * @param string $dbName
     * @param string $dbUserName
     * @param string $dbPassword
     *
     * @return bool
     */
    public static function checkDbConnection(
        string $dbConnection,
        string $dbHost,
        string $dbPort,
        string $dbName,
        string $dbUserName,
        string $dbPassword
    )
    {
        if (!DatabaseChecker::isDbValid($dbConnection, $dbHost, $dbPort, $dbName, $dbUserName, $dbPassword)) {
            return false;
        }

        // Set database details
        static::saveDbVariables($dbConnection, $dbHost, $dbPort, $dbName, $dbUserName, $dbPassword);

        // Try to increase the maximum execution time
        set_time_limit(300); // 5 minutes

        // Create tables

        // Passport migrations
        Artisan::call('migrate:fresh', [
            '--path' => 'vendor/laravel/passport/database/migrations',
            '--force' => true,
        ]);

        // All other tables
        Artisan::call('migrate', [
            '--force' => true,
        ]);

        Artisan::call('passport:api', [
            '--force' => true,
        ]);

        // Create Roles and Permissions
        Artisan::call('db:seed', [
            '--class' => 'PermissionsWithWeightsSeeder',
            '--force' => true,
        ]);

        Artisan::call('db:seed', [
            '--class' => 'RolesAndPermissionsSeeder',
            '--force' => true,
        ]);

        Artisan::call('db:seed', [
            '--class' => 'TemplatesStylesTableSeeder',
            '--force' => true,
        ]);

        return true;
    }

    public static function saveDbVariables($dbConnection, $host, $port, $database, $username, $password)
    {
        // Update .env file
        static::updateEnv([
            'DB_CONNECTION' =>  $dbConnection,
            'DB_HOST'       =>  $host,
            'DB_PORT'       =>  $port,
            'DB_DATABASE'   =>  $database,
            'DB_USERNAME'   =>  $username,
            'DB_PASSWORD'   =>  $password,
        ]);

        $connection = env('DB_CONNECTION', 'mysql');

        // Change current connection
        $db = Config::get('database.connections.' . $connection);

        $db['host'] = $host;
        $db['database'] = $database;
        $db['username'] = $username;
        $db['password'] = $password;

        Config::set('database.connections.' . $connection, $db);

        DB::purge($connection);
        DB::reconnect($connection);
    }

    /**
     * @param string $name
     * @param string $address
     * @param string $sipServerAddress
     * @param string|null $sipServerPort
     * @param string $conciergeSipNumber
     *
     * @return bool
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public static function createFirstObject(
        string $name,
        string $address,
        string $sipServerAddress,
        $sipServerPort,
        string $conciergeSipNumber
    )
    {
        $objectModel = ObjectModel::first();
        if (!$objectModel) {
            $objectModel = new ObjectModel;
        }

        $attributes = [
            $objectModel->getKeyName() => $objectModel->getKey(),
            'name' => $name,
            'address' => $address,
            'sip_server_address' => $sipServerAddress,
            'sip_server_port' => $sipServerPort,
            'concierge_sip_number' => $conciergeSipNumber,
        ];
        foreach ($attributes as $attribute => $value) {
            if (in_array($attribute, $objectModel->translatable)) {
                $translations = [];

                foreach (config('app.locales') as $locale) {
                    $translations[$locale] = $value;
                }

                $attributes[$attribute] = json_encode($translations);
            }
        }

        $objectModel->setRawAttributes($attributes);

        DB::beginTransaction();

        if ($objectModel->save()) {
            $acsHttpRequestHelper = new AcsHttpRequestHelper();
            $response = $acsHttpRequestHelper->sendPost('/api/object', [
                'id' => (string)$objectModel->id,
                'name' => $objectModel->getTranslations('name'),
                'address' => $objectModel->getTranslations('address'),
                'sip_server_address' => $objectModel->sip_server_address,
                'sip_server_port' => $objectModel->sip_server_port,
                'concierge_sip_number' => $objectModel->concierge_sip_number,
            ]);

            if (isset($response['status']) && ($response['status'] === 200)) {
                DB::commit();

                // Set object cookie
                Cookie::queue(Cookie::forever('object', (int)$objectModel->id));

                // Update .env file
                static::updateEnv([
                    'APP_FIRST_OBJECT_CREATED' => 'true',
                ]);

                return true;
            } else {
                DB::rollBack();
            }
        }

        return false;
    }

    /**
     * @param string $name
     * @param string $email
     * @param string $password
     *
     * @return bool
     */
    public static function createAdmin(
        string $name,
        string $email,
        string $password
    )
    {
        $admin = User::firstOrCreate([
            'email' => $email,
        ], [
            'name' => $name,
            'password' => $password,
        ]);

        if ($admin) {
            $admin->givePermissionTo(Permission::FULL_ACCESS);

            return true;
        }

        return false;
    }

    public static function laravelConfig()
    {
        // Fill up settings table
        $defaultSettings = (new \SettingsTableSeeder)->getSettings();

        foreach ($defaultSettings as $defaultSetting) {
            if ($defaultSetting['key'] == 'active_locales') {
                // Removing Ukrainian locale from "Active locales"
                // Ukrainian is available only in some APIs
                $defaultSetting['value'] = implode(',', array_diff(config('app.locales'), ['ua']));
            }

            $settingExist = DB::table('settings')->where('key', $defaultSetting['key'])->exists();

            if($settingExist) {
                DB::table('settings')->where('key', $defaultSetting['key'])->update($defaultSetting);
            } else {
                DB::table('settings')->insert($defaultSetting);
            }
        }

        // Update .env file
        static::updateEnv([
            'APP_INSTALLED' => 'true',
            'APP_INSTALLATION_DATETIME' => date('"d.m.Y H:i:s"'),
        ]);
    }
}
