<?php

namespace evApps\Installer\app\Http\Utilities;

use App\Helpers\PHPini;

/**
 * Class ServerRequirementChecker
 *
 * @package evApps\Installer\app\Http\Utilities
 */
class ServerRequirementChecker
{

    const MIN_UPLOAD_FILE_SIZE = 70 * 1024 * 1024;
    const MAX_EXECUTION_TIME = 60 * 60;
    const MIN_POST_MAX_SIZE = 70 * 1024 * 1024;

    /**
     * @var array $requirements
     */
    protected static $requirements = [
        'php_version' => '7.0.0',
        'php_extension_openssl',
        'php_extension_pdo',
        'php_extension_mbstring',
        'php_extension_tokenizer',
        'php_extension_xml',
        'php_extension_curl',
        'writable_.env',
        'writable_bootstrap/cache',
        'writable_storage',
        'min_upload_file_size' => self::MIN_UPLOAD_FILE_SIZE,
        'min_post_max_size' => self::MIN_POST_MAX_SIZE,
        'max_execution_time' => self::MAX_EXECUTION_TIME,
        'max_php_version' => '7.3.0'
    ];


    /**
     * @return bool
     */
    public static function checkPhpVersion()
    {
        return version_compare(PHP_VERSION, self::$requirements['php_version'], '>=') > 0 && version_compare(PHP_VERSION, self::$requirements['max_php_version'], '<') > 0;
    }

    /**
     * @param string $extension
     *
     * @return bool
     */
    public static function checkPhpExtension(string $extension)
    {
        return extension_loaded($extension);
    }

    /**
     * @param string $file
     *
     * @return bool
     */
    public static function checkFileIsWritable(string $file)
    {
        return is_writable(base_path($file));
    }

    /**
     * @param int|null $size
     *
     * @return bool
     */
    public static function checkMinUploadFileSize(int $size = null)
    {
        $fileUploadMaxSize = $size ?? PHPini::file_upload_max_size();

        return $fileUploadMaxSize >= self::$requirements['min_upload_file_size'];
    }

    public static function checkMinPostMaxSize(int $size = null)
    {
        $fileUploadMaxSize = $size ?? PHPini::post_max_size();

        return $fileUploadMaxSize >= self::$requirements['min_post_max_size'];
    }

    /**
     * @param int|null $time
     *
     * @return bool
     */
    public static function checkMaxExecutionTime(int $time = null)
    {
        $maxExecutionTime = $time ?? ini_get('max_execution_time');

        return $maxExecutionTime >= self::$requirements['max_execution_time'];
    }

    public static function checkAll()
    {
        $checkingResults = [];

        foreach (self::$requirements as $requirement => $value) {
            if ($requirement === 'php_version') {
                $checkingResults['php_version'] = self::checkPhpVersion();
            } elseif ($requirement === 'min_upload_file_size') {
                $checkingResults['min_upload_file_size'] = self::checkMinUploadFileSize();
            } elseif ($requirement === 'max_execution_time') {
                $checkingResults['max_execution_time'] = self::checkMaxExecutionTime();
            } elseif (strpos($value, 'php_extension_') === 0) {
                $checkingResults[$value] = self::checkPhpExtension(str_replace('php_extension_', '', $value));
            } elseif (strpos($value, 'writable_') === 0) {
                $checkingResults[$value] = self::checkFileIsWritable(str_replace('writable_', '', $value));
            } elseif ($requirement === 'min_post_max_size') {
                $checkingResults['min_post_max_size'] = self::checkMinPostMaxSize();
            }
        }

        return $checkingResults;
    }
}
