<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Backpack\CRUD\CrudTrait;
use Illuminate\Support\Facades\Auth;
use Illuminate\Database\Eloquent\Builder;

/**
 * Class Quiz
 *
 * @property int    $object_id
 * @property string $title
 * @property string $question
 * @property int    $free_answer
 * @property int    $multiple
 * @property int    $allow_view_results
 * @property string $start_at
 * @property string $end_at
 * @property int    $is_filled
 * @property int    $is_private
 *
 * @property \App\Models\Object $object
 *
 * @package App\Models
 *
 * @mixin \Illuminate\Database\Eloquent\Model
 */
class Quiz extends Model
{
    use CrudTrait;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'quizzes';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'object_id',
        'title',
        'question',
        'free_answer',
        'multiple',
        'allow_view_results',
        'start_at',
        'end_at',
        'is_filled',
        'is_private',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array
     */
    protected $hidden = [
        'created_at',
        'updated_at',
        'is_template',
        'category',
        'is_filled',
        'push_sent',
        'tags',
    ];

    /**
     * Indicates if the model should be timestamped.
     *
     * @var bool
     */
    public $timestamps = true;

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    protected static function boot()
    {
        parent::boot();

        static::addGlobalScope('template', function (Builder $builder) {
            $builder->where('is_template', '=', 0);
        });

    }

    // relations
    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function object()
    {
        return $this->belongsTo(Object::class, 'object_id', 'id');
    }

    public function answers()
    {
        return $this->hasMany('App\Models\QuizAnswer', 'quiz_id', 'id');
    }

    public function destinations()
    {
        return $this->hasMany('App\Models\QuizDestination', 'quiz_id', 'id');
    }

    public function results()
    {
        return $this->hasMany('App\Models\QuizResult', 'quiz_id', 'id');
    }

    public function resultsFree()
    {
        return $this->hasMany('App\Models\QuizResult', 'quiz_id', 'id')
            ->where('free_answer_value', '!=', null);
    }

    public function selectedAnswer()
    {
        return $this->hasMany('App\Models\QuizAnswer', 'quiz_id', 'id')
            ->whereHas('results', function (Builder $query){
                $query->where('user_id', Auth::user()->id);
            });

    }

    public function getWhom()
    {
        $id = $this->id;
        return $this->getDestinations($id);
    }

    public function getDestinations($id)
    {
        $destinations = QuizDestination::where('quiz_id', $id)->get();
        if (count($destinations) == 0) {
            return '';
        }
        if ($destinations[0]->is_all_house) {
            return __('adverts.sent_result.is_all_house');
        }
        $recipients = [];
        foreach ($destinations as $destination) {
            if ($destination->entrance) {
                $recipients[] = __('apartments.crud_fields.entrance').' '.$destination->entrance;
            }
            if ($destination->floor) {
                $recipients[] = __('apartments.crud_fields.floor').' '.$destination->floor;
            }
            if ($destination->apartment) {
                $recipients[] = __('apartments.apartment').' '.$destination->apartment;
            }
        }
        return implode(', ', $recipients);
    }

    public function selectedFree()
    {
        return $this->hasOne('App\Models\QuizResultFree', 'quiz_id', 'id')
            ->where('user_id', Auth::user()->id);
    }

    public function viewed()
    {
        return $this->hasMany('App\Models\QuizViewed', 'quiz_id', 'id');
    }

    // mutators

    public function getStatusAttribute()
    {
        $start_at = (new \DateTime($this->start_at))->getTimestamp();
        $current = (new \DateTime(date('Y-m-d H:i:s')))->add(new \DateInterval('P0DT3H'))->getTimestamp();
        $end_at = (new \DateTime($this->end_at))->getTimestamp();

        if (($start_at <= $current && $end_at > $current) && $this->is_filled == 0) {
            return __('quizzes.statuses.in_process');
        } elseif ($end_at <= $current || $this->is_filled == 1) {
            return __('quizzes.statuses.finished');
        } else {
            return __('quizzes.statuses.not_published');
        }
    }

    public function getStatus()
    {
        if (($this->start_at <= date('Y-m-d H:i:s') && $this->end_at > date('Y-m-d H:i:s')) && $this->is_filled == 0) {
            return 0;
        } elseif ($this->end_at <= date('Y-m-d H:i:s') || $this->is_filled == 1) {
            return 1;
        } else {
            return 2;
        }
    }


    /**
     * Является ли заявка действующей
     *
     * @return boolean
     */
    public function isActive()
    {
        $date = date('Y-m-d H:i:s');

        if ($this->start_at <= $date && $date <= $this->end_at) {
            return true;
        }

        return false;
    }


    /**
     * Будет ли заявка действительна в будущем
     *
     * @return boolean
     */
    public function isWillActive()
    {
        $date = date('Y-m-d H:i:s');

        if ($this->start_at > $date) {
            return true;
        }

        return false;
    }
}
