<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Backpack\CRUD\CrudTrait;
use Carbon\Carbon;
use Illuminate\Support\Facades\App;

class Apartment extends Model
{
    use CrudTrait;

     /*
    |--------------------------------------------------------------------------
    | GLOBAL VARIABLES
    |--------------------------------------------------------------------------
    */

    protected $table = 'apartments';
    protected $primaryKey = 'id';
    // public $timestamps = false;
    // protected $guarded = ['id'];
    protected $fillable = ['number', 'entrance', 'floor', 'activation_code', 'activation_limit', 'object_id', 'is_local', 'is_save'];
    protected $hidden = ['activation_code', 'is_local', 'is_save'];
    protected $export = false;
    // protected $dates = [];

    /*
    |--------------------------------------------------------------------------
    | FUNCTIONS
    |--------------------------------------------------------------------------
    */

    /**
     * Возвращает количество устройств у всех пользователей в квартире
     *
     * @return int
     */
    public function getActivationsUsed(): int
    {
        return $this
          ->users()
          ->withCount('tokens')
          ->pluck('tokens_count')
          ->sum();
    }

    public static function getMaxSipNumbers()
    {
        return env('MAX_SIP_NUMBERS', 5);
    }

    public function getSipNumbersHtml(int $limit = 0)
    {
        $sipNumbers = $this->sip_numbers->pluck('id')->toArray();

        $connectedDevices = Device::whereIn('sip_number_id', $sipNumbers)->pluck('sip_number_id')->toArray();

        $template = '';
        if ($sipNumbers) {
            // Определние максимального количества выводимых sip номеров
            $sipNumbers_count = count($sipNumbers);
            if ($limit && $limit <= $sipNumbers_count) {
                $sipNumbers_count = $limit;
            }

            $template .='<span class="sip-numbers-used">';
            for ($i = 0; $i < $sipNumbers_count; $i++) {
                $apartment_sip_number = ApartmentSipNumber::find($sipNumbers[$i]);
                $sip_number = null;
                if ($apartment_sip_number instanceof ApartmentSipNumber) {
                    $sip_number = $apartment_sip_number->sip_number;
                }

                if (in_array($sipNumbers[$i], $connectedDevices)) {
                    $template .= '<span class="sip-is-used" title="'.__('apartments.crud_fields.sip_activated').'">'.$sip_number.'</span>, ';
                } else {
                    $template .= $sip_number.', ';
                }
            }

            $template = substr($template,0,-2);

            if (count($sipNumbers) > $limit && $limit !== 0) {
                $template .= '...';
            }

            $template .='</span>';
        }

        return $template;
    }

    public function getSipNumbersHtmlForTable()
    {
        return $this->getSipNumbersHtml(5);
    }

    public function getActivationsUsedHtml()
    {
        $used = $this->getActivationsUsed();
        return '<span class="activations-used-value" data-id="'.$this->id.'">' . $used . '</span>';
    }

    public function getFloors()
    {
        $floors = self::select('floor')->where('object_id', '=', config('app.object.id'))->distinct()->get();

        return $floors->mapWithKeys(function ($item) {
            return [$item['floor'] => __('apartments.crud_fields.floor') . ' ' . $item['floor']];
        })->toArray();
    }

    public static function getObjectFloors()
    {
        $floors = self::select('floor')->where('object_id', '=', config('app.object.id'))->distinct()->get();

        return $floors->mapWithKeys(function ($item) {
            return [$item['floor'] => __('apartments.crud_fields.floor') . ' ' . $item['floor']];
        })->toArray();
    }

    public function getEntrances()
    {
        $entrances = self::select('entrance')->where('object_id', '=', config('app.object.id'))->distinct()->get();

        return $entrances->mapWithKeys(function ($item) {
            return [$item['entrance'] => __('apartments.crud_fields.entrance') . ' ' . $item['entrance']];
        })->toArray();
    }

    public static function getObjectEntrances()
    {
        $entrances = self::select('entrance')->where('object_id', '=', config('app.object.id'))->distinct()->get();

        return $entrances->mapWithKeys(function ($item) {
            return [$item['entrance'] => __('apartments.crud_fields.entrance') . ' ' . $item['entrance']];
        })->toArray();
    }

    public function getFlats($entrance = NULL, $floor = NULL) {
        $flats = $this->where('object_id', '=', config('app.object.id'));

        if (is_int($floor) && $floor >= 0) {
            $flats->where('floor', '=', $floor);
        }

        if (is_int($entrance) && $entrance >= 0) {
            $flats->where('entrance', '=', $entrance);
        }

        return $flats->get()->mapWithKeys(function ($item) {
            return [$item['number'] => __('apartments.apartment') . ' ' . $item['number']];
        })->toArray();
    }

    public static function getObjectFlats($entrance = NULL, $floor = NULL)
    {
        $flats = self::select('number')->where('object_id', '=', config('app.object.id'))->distinct()->get();

        return $flats->mapWithKeys(function ($item) {
            return [$item['number'] => __('apartments.crud_fields.number') . ' ' . $item['number']];
        })->toArray();
    }

    public function getHouseTree() {
        $tree = [];

        $apartments = Apartment::select('id', 'number', 'entrance', 'floor')
            ->where('object_id', config('app.object.id'))
            ->orderByRaw(
                'number+0,'.
                'number,'.
                'LENGTH(\'number\')'
            )
            ->get()->toArray();

        foreach($apartments as $ap){
            $tree[$ap['entrance']]['name'] = __('apartments.crud_fields.entrance') . ' ' . $ap['entrance'];
            $tree[$ap['entrance']]['items'][$ap['floor']]['name'] = __('apartments.crud_fields.floor') . ' ' . $ap['floor'];
            $tree[$ap['entrance']]['items'][$ap['floor']]['items'][$ap['id']] = [
                'name' => __('apartments.apartment') . ' ' . $ap['number'],
                'number' => $ap['number'],
                'id' => $ap['id']
            ];
        }

        $this->keysSort($tree);

        return $tree;
    }

    public function getHouseTreeById() {
        $tree = [];

        $apartments = Apartment::select('id', 'number', 'entrance', 'floor')
            ->where('object_id', config('app.object.id'))
            ->orderByRaw(
                'number+0,'.
                'number,'.
                'LENGTH(\'number\')'
            )
            ->get()->toArray();

        foreach($apartments as $ap){
            $tree[$ap['entrance']]['name'] = __('apartments.crud_fields.entrance') . ' ' . $ap['entrance'];
            $tree[$ap['entrance']]['items'][$ap['floor']]['name'] = __('apartments.crud_fields.floor') . ' ' . $ap['floor'];
            $tree[$ap['entrance']]['items'][$ap['floor']]['items'][$ap['id']] = [
                'name' => __('apartments.apartment') . ' ' . $ap['number'],
                'number' => $ap['number'],
                'id' => $ap['id']
            ];
        }

        $this->keysSort($tree);

        return $tree;
    }

    private function keysSort(&$tree)
    {
        // Сортировка массива по ключам
        ksort($tree);

        // Сортировка этажей в полученном массиве
        foreach ($tree as &$entrance) {
            ksort($entrance['items']);
        } unset($entrance);

        return $tree;
    }


    /*
    |--------------------------------------------------------------------------
    | RELATIONS
    |--------------------------------------------------------------------------
    */

    public function object() {
        // it's strange, but without 'foreignKey' and 'ownerKey' params Relation doesn't worked
        return $this->belongsTo(Building::class, 'object_id', 'id');
    }

    public function users()
    {
        return $this->belongsToMany(User::class);
    }

    public function sip_numbers()
    {
        return $this->hasMany('App\Models\ApartmentSipNumber', 'apartment_id', 'id');
    }

    public function cameras()
    {
        return $this->belongsToMany('App\Models\Camera');
    }

    public function calling_panels()
    {
        return $this->belongsToMany('App\Models\CallingPanel');
    }
    /*
    |--------------------------------------------------------------------------
    | SCOPES
    |--------------------------------------------------------------------------
    */

    /*
    |--------------------------------------------------------------------------
    | ACCESORS
    |--------------------------------------------------------------------------
    */



    /*
    |--------------------------------------------------------------------------
    | MUTATORS
    |--------------------------------------------------------------------------
    */

    public function getPrepareCodeHtml()
    {
      if($this->activation_code){
        $result = str_split($this->activation_code, 4);
        return '<span style="white-space: nowrap;">'.$result[0].'-'.$result[1].'-'.$result[2].'-'.$result[3].'</span>';
      } else {
        return trans('apartments.generating');
      }
    }

    public function getPrepareActivationHtml()
    {
      return '<span style="white-space: nowrap;">' . $this->getActivationsUsedHtml() .' '. trans('apartments.from') . ' ' . $this->activation_limit .'</span>';
    }

    public static function getGroupByHouse ()
    {
        $apartments = (new static)->where('is_local', 1)->where('object_id', config('app.object.id'))->get();
        $apartments_groups = [];

        foreach ($apartments as $apartment) {
            $apartments_groups[$apartment->entrance][$apartment->floor][] = $apartment;
        }

        return $apartments_groups;
    }

    public static function removeLocal() {
        $object_id = config('app.object.id');
        (new static)->where('is_local', 1)->where('is_save', 0)->where('object_id', $object_id)->delete();
    }

    public static function checkNameDuplicatesInGrouping ($groupByHouse)
    {
        $duplicates_items = [];
        $numbers = [];

        if(!empty($groupByHouse)) {
            foreach ($groupByHouse as $entrance_number => $entrance) {
                if (!empty($entrance)) {
                    foreach ($entrance as $floor_number => $floor) {
                        if (!empty($floor)) {
                            foreach ($floor as $apartment_id => $apartment) {
                                if(!empty($apartment['number'])) {
                                    $duplicate_key = array_search($apartment['number'], $numbers);

                                    if($duplicate_key !== false) {
                                        $duplicates_items[$apartment_id] = $apartment['number'];
                                        $duplicates_items[$duplicate_key] = $numbers[$duplicate_key];
                                    }
                                }
                                $numbers[$apartment_id] = $apartment['number'];
                            }
                        }
                    }
                }
            }
        }

        return $duplicates_items;
    }

    public static function getGroupByHouseInfo($groupByHouse)
    {
        $info = [
            'number_entrances' => 0,
            'number_floors' => 0,
            'number_apartments_per_floors' => 0,
            'number_apartments' => 0,
            'activation_limit' => 0,
            'required_number_activations' => 0
        ];

        if(!empty($groupByHouse)) {
            foreach ($groupByHouse as $entrance_number => $entrance) {
                if (!empty($entrance)) {
                    $info['number_entrances']++;
                    $info['number_floors'] = 0;
                    foreach ($entrance as $floor_number => $floor) {
                        if (!empty($floor)) {
                            $info['number_floors']++;
                            $info['number_apartments_per_floors'] = 0;
                            foreach ($floor as $apartment_id => $apartment) {
                                $info['number_apartments']++;
                                $info['required_number_activations'] += intval($apartment->activation_limit);
                                $info['activation_limit'] = intval($apartment->activation_limit);
                                $info['number_apartments_per_floors']++;
                            }
                        }
                    }
                }
            }
        }

        return $info;
    }

    public static function insertOrUpdateApartments(array $rows){
        $table = \DB::getTablePrefix().with(new self)->getTable();


        $first = reset($rows);

        $columns = implode( ',',
            array_map( function( $value ) { return "$value"; } , array_keys($first) )
        );

        $values = implode( ',', array_map( function( $row ) {
                return '('.implode( ',',
                        array_map( function( $value ) { return '"'.str_replace('"', '""', $value).'"'; } , $row )
                    ).')';
            } , $rows )
        );

        $updates = implode( ',',
            array_map( function( $value ) { return "$value = VALUES($value)"; } , array_keys($first) )
        );
//      Да, здесь используется именно такой способ вставки/обноления данных, т.к. позволяет вставить/обновить записи в таблице массово,
//      избежать хождения по циклу и вставлять/обновлять каждую строку отдельно (это позвооляет ускорить генерацию квартир на более чем на 25с)
//      Метод laravel updateOrCreate работает только с одной записью
        $sql = "INSERT INTO {$table}({$columns}) VALUES {$values} ON DUPLICATE KEY UPDATE {$updates}";

        return \DB::statement( $sql );
    }

    public function getLastActivity()
    {
        //TODO Установку временной зоны нужно вынести в настройки. Здесь и в других шаблонах, где встречается принудительная установка timezone
        $last_activity = ActivityLog::where('apartment_id', $this->id)
            ->orderByDesc('updated_at')->select('updated_at')->first();
        return $last_activity ?
            $last_activity->updated_at
                ->setTimezone('Europe/Moscow')
                ->format('d.m.Y H:i')
            : "";
    }

    public function getActivityFourWeeksAgo()
    {
        return $this->checkingWeekActivity(21);
    }

    public function getActivityThreeWeeksAgo()
    {
        return $this->checkingWeekActivity(14);
    }

    public function getActivityTwoWeeksAgo()
    {
        return $this->checkingWeekActivity(7);
    }

    public function getActivityOneWeekAgo()
    {
        return $this->checkingWeekActivity(0);
    }

    private function checkingWeekActivity($number)
    {
        $date = Carbon::createMidnightDate();
        $delta = 3; // TODO Установить разницу между utc и часовым поясом сервера
        if ($this->checkingLogInfoExistence($date->subDays($number)->subHours($delta))) {
            $start = clone($date)->subDays(7);
            $end = clone($date)->addDays(7);
            return $this->checkingActivityBetween($start, $end);
        }
        return $this->greyCircleHtml();
    }

    private function checkingActivityBetween($start, $end)
    {
        $activity = ActivityLog::where('apartment_id', $this->id)
            ->where('updated_at', '<', $end)
            ->where('updated_at', '>=', $start)
            ->get()->toArray();
        if (empty($activity)) {
            return $this->redCircleHtml();
        } else {
            return $this->greenCircleHtml();
        }
    }

    private function checkingLogInfoExistence($end)
    {
        $apartmentExist = ActivityLog::where('apartment_id', $this->id)->first();
        if (!$apartmentExist) {
            return false;
        }
        $activity = ActivityLog::where('apartment_id', $this->id)
            ->where('created_at', '<', $end)
            ->get()->toArray();
        return !empty($activity);
    }

    public function setExport()
    {
        $this->export = true;
    }

    private function greyCircleHtml()
    {
        return $this->export ?
            "FF555555"
            : "<span class='activity_circle'><i class='fa fa-circle' style='color:#555'></i></span>";
    }

    private function redCircleHtml()
    {
        return $this->export ?
            "FFCC0000"
            : "<span class='activity_circle'><i class='fa fa-circle' style='color:#c00'></i></span>";
    }

    private function greenCircleHtml()
    {
        return $this->export ?
            "FF008800"
            : "<span class='activity_circle'><i class='fa fa-circle' style='color:#080'></i></span>";
    }
}
