<?php

namespace App\Http\Requests;

use App\Http\Requests\Request;
use App\Models\CallingPanel;
use App\Rules\DomainNameOrIp;
use App\Rules\Rtsp\Rtsppassword;
use App\Rules\Rtsp\Rtspurl;
use App\Rules\Rtsp\Rtspname;

class CallingPanelRequest extends \Backpack\CRUD\app\Http\Requests\CrudRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        // only allow updates if the user is logged in
        return \Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rtsp = [
            'link' => new Rtspurl,
            'name' =>  new Rtspname,
            'password' => new Rtsppassword,
            'domain_name_or_ip' => new DomainNameOrIp,
            'port' => 'integer|min:1|'
        ];

        $rtsp_link_type = $this->request->get('rtsp_link_type');

        switch ($rtsp_link_type) {
            case CallingPanel::RTSP_LINK_TYPE_FULL:
                $rtsp = [
                    'link' => new Rtspurl,
                    'name' =>  null,
                    'password' => null,
                    'domain_name_or_ip' => null,
                    'port' => null
                ];
                break;

            case CallingPanel::RTSP_LINK_TYPE_PARAMETERS:
                $rtsp = [
                    'link' => null,
                    'name' =>  new Rtspname,
                    'password' => new Rtsppassword,
                    'domain_name_or_ip' => new DomainNameOrIp,
                    'port' => 'integer|min:1|'
                ];
                break;
        }

        return [
            'name' => 'required|string',
            'sip_number' => 'sometimes|string',
            'rtsp_link' => ['nullable', 'required_if:rtsp_link_type,==,' . CallingPanel::RTSP_LINK_TYPE_FULL, $rtsp['link']],
            'login' => ['nullable', 'required_if:rtsp_link_type,==,' . CallingPanel::RTSP_LINK_TYPE_PARAMETERS, $rtsp['name']],
            'password' => ['nullable', 'required_if:rtsp_link_type,==,' . CallingPanel::RTSP_LINK_TYPE_PARAMETERS, $rtsp['password']],
            'ip_address' => ['required_if:rtsp_link_type,==,' . CallingPanel::RTSP_LINK_TYPE_PARAMETERS, 'nullable', $rtsp['domain_name_or_ip']],
            'port' => 'nullable|'.$rtsp['port'].'required_if:rtsp_link_type,==,' . CallingPanel::RTSP_LINK_TYPE_PARAMETERS,
            'dtmf_1' => 'required|nullable|string',
            'dtmf_2' => 'nullable|string|required_if:has_second_relay,==,1',
        ];
    }

    /**
     * Get the validation attributes that apply to the request.
     *
     * @return array
     */
    public function attributes()
    {
        $attributes_names = [
            'name',
            'sip_number',
            'rtsp_link',
            'login',
            'password',
            'ip_address',
            'port',
            'dtmf_1',
            'dtmf_2',
        ];

        $attributes = [];

        foreach ($attributes_names as $attribute_name) {
            $attributes[$attribute_name] = __("calling_panels.crud_fields.$attribute_name");
        }

        return $attributes;
    }

    /**
     * Get the validation messages that apply to the request.
     *
     * @return array
     */
    public function messages()
    {
        return [
            //
        ];
    }

    public function preProcess()
    {
        if ($this->input('rtsp_link_type') == CallingPanel::RTSP_LINK_TYPE_FULL) {
            $this->request->set('login', null);
            $this->request->set('password', null);
            $this->request->set('ip_address', null);
            $this->request->set('port', null);
        } else {
            $rtsp= 'rtsp://' . $this->request->get('login')
                . ':' . $this->request->get('password')
                . '@' . $this->request->get('ip_address')
                . ':' . $this->request->get('port');
            $this->request->set('rtsp_link', $rtsp);
        }

        if ($this->input('has_second_relay') == 0) {
            $this->request->set('dtmf_2', null);
        }
    }
}
