<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Notice;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Config;

class NoticeController extends Controller {

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response | array
     */
    public function index() {

        $user = Auth::user();
        $userId = Auth::user()->id;
        $userApartments = Auth::user()->apartments()->get()->pluck('number');
        $archiveTime = Config::get('settings.archival_time');
        $archiveTime = !empty($archiveTime) ? $archiveTime : 3;
        $archiveDate = Carbon::today()->subMonths($archiveTime);

        // Get User device: this is workaround for getting user's device.
        // See \App\Models\User::device() relation
        $device = $user->device();

        if ($device) {
            $device = $device->first();
        }

        if ($device && $device->apartment) {
            $objectId = $device->apartment->object->id;
            $userId = $user->id;
            $userApartments = $user->apartments;

            $advertsRaw = Notice::whereIn('apartment', $userApartments->pluck('number'))
                    ->where(function (Builder $query) use ($objectId, $archiveDate) {
                        $query->where('notice.object_id', '=', $objectId);
                        $query->whereDate('notice.actual_to_date', '>=', $archiveDate);
                    })
                    ->selectRaw('`id`, `theme`, `text`, `actual_from_date`, `actual_to_date`, IF(`notice_viewed`.`notice_id` IS NULL, FALSE, TRUE) as `is_viewed`')
                    ->distinct(['notice.id'])
                    ->leftJoin('notice_viewed', function ($join) use ($userId) {
                        $join
                            ->on('notice_viewed.notice_id', '=', 'notice.id')
                            ->on('notice_viewed.user_id', '=', \DB::raw($userId));
                    })
                    ->orderBy('is_viewed')
                    ->orderByDesc('actual_from_date')
                    ->orderByDesc('created_at')
                    ->get('id');
                    
            /*
             * Сортировка
             * - непрочитанные активные,
             * - прочитанные активные,
             * - непрочитанные архивные,
             * - прочитанные архивные.
             * */
            $advertsGroups = [[], [], [], []];
            foreach($advertsRaw as &$item) {
                if(Carbon::parse($item->actual_to_date) >= Carbon::now()) {
                    if(empty($item->is_viewed)) {
                        $advertsGroups[0][] = $item; // непрочитанные активные
                    }  else {
                        $advertsGroups[1][] = $item; // непрочитанные активные
                    }
    
                } else {
                    if(empty($item->is_viewed)) {
                        $advertsGroups[2][] = $item; // непрочитанные архивные,
                    }  else {
                        $advertsGroups[3][] = $item; // прочитанные архивные.
                    }
                }
            }
    
            $advertsSortRaw = [];
    
            foreach($advertsGroups as $key => $advertsGroup) {
                $advertsSortRaw = array_merge($advertsSortRaw, $advertsGroup);
            }
    
            return $advertsSortRaw;
        }

        return [];
    }

    /**
     * @param int $noticeId
     *
     * @return \App\Models\Notice
     */
    public function markAsRead($noticeId)
    {
        try {
            $notice = Notice::findOrFail($noticeId);
        } catch (\Exception $e) {
            return response()->json(['message' => trans('notice.notice_not_found')], 404);
        }

        $data = [
            'user_id' => Auth::user()->id,
            'notice_id' => $notice->id,
        ];

        $isNotViewed = Validator::make($data, [
            'notice_id' => 'unique_with:notice_viewed,user_id',
        ])->passes();

        if ($isNotViewed) {
            $notice->viewed()->create($data);
        }

        $notice->is_viewed = 1;

        return $notice;
    }
}
