<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Claim;
use App\Models\ClaimType;
use App\Models\User;
use App\Models\Apartment;
use App\Models\ClaimImage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
//use PHPini;

class ClaimController extends Controller {

    protected $rules = [
        'need_at' => 'required|date|after:now',
        'type' => 'required|numeric|isexists:claim_type,id',
        'phone' => 'required',
        'claim_text' => 'string|max:1000'
    ];

    /**
     * Claim types
     */
    public function types() {
        return ClaimType::all();
    }

    /**
     * @param \Illuminate\Http\Request $request
     *
     * @return \App\Models\Claim|\Illuminate\Http\JsonResponse
     */
    public function createClaim(Request $request) {
        $this->validate($request, $this->rules);

        $claim = new Claim;
        $claim->need_at = $request->get('need_at');
        $claim->type_id = $request->get('type');
        $claim->phone = $this->clearPhone($request->get('phone'));
        $claim->claim_text = $request->get('claim_text');
        $claim->status = 0;

        $user_id = Auth::user()->id;
        $apartments = User::findOrFail($user_id)
            ->apartments()
            ->pluck('id')
            ->toArray();

        if (!$apartments) {
            return response()->json(['message' => trans('claims.validator.apartment_not_found')], 404);
        }
        if (count($apartments) > 1) {
            return response()->json(['message' => trans('claims.validator.multiple_apartments')], 300);
        }

        $apartment = Apartment::findOrFail($apartments[0]);

        $claim->apartment_id = $apartment->id;
        $claim->object_id = $apartment->object->id;
        $claim->user_id = $user_id;

        $claim->save();

        return $claim;
    }

    /**
     * Обновление заявки
     */
    public function updateClaim(Request $request, $id) {
        $this->validate($request, $this->rules);

        try {
            $claim = Claim::with('images')->findOrFail($id);
        } catch (\Exception $e) {
            return response()->json(['message' => trans('claims.claim_not_found')], 404);
        }
        $user = Auth::user();
        if ($user->id != $claim->user_id) {
            return response()->json(['message' => trans('claims.claim_of_another_user')], 403);
        }
        if ($claim->status != 0) {
            return response()->json(['message' => trans('claims.claim_cannot_be_changed')], 403);
        }
        $claim->need_at = $request->get('need_at');
        $claim->type_id = $request->get('type');
        $claim->claim_text = $request->get('claim_text');
        $claim->phone = $this->clearPhone($request->get('phone'));

        $claim->save();

        return $claim;

    }

    /**
     * Получение одной заявки по id
     */
    public function getClaim($claimId) {
        try {
            $claim = Claim::findOrFail($claimId);
        } catch (\Exception $e) {
            return response()->json(['message' => trans('claims.claim_not_found')], 404);
        }

        $claim->type = $claim->getType($claim->type_id);
        $claim->status = Claim::claimStatuses()[$claim->status];
        $claim->load('images');

        return $claim;
    }

    /**
     * Получение всех заявок пользователя
     */
    public function getClaims(Request $request) {
        $userId = \Auth::user()->id;
        $claims = Claim::where('user_id', $userId)->orderByDesc('id')->with('images')->get();
        $statuses = Claim::claimStatuses();

        foreach ($claims as &$claim) {
            $claim->type = $claim->getType($claim->type_id);
            $claim->status = $statuses[$claim->status];
        }

        return $claims;
    }

    /**
     * Добавляет изображение если есть и возвращает поля заявки
     */
    public function images(Request $request, $id) {
        try {
            $claim = Claim::with('images')->findOrFail($id);
        } catch (\Exception $e) {
            return response()->json(['message' => trans('claims.claim_not_found')], 404);
        }

        $user = Auth::user();

        if ($user->id != $claim->user_id) {
            return response()->json(['message' => trans('claims.claim_of_another_user')], 200);
        }

        if ($claim->status != 0) {
            return response()->json(['message' => trans('claims.claim_cannot_be_changed')], 200);
        }

        // Файлы есть
        if(count($request->file()) > 0) {
            $image_errors = [];
            $max_claim_images_count = 3;
            $claim_images_count = $claim->images()->count();
            $add_images_count = $max_claim_images_count - $claim_images_count;

            if($add_images_count > 0) {
                $files = array_slice($request->file(), 0, $add_images_count, true);

                $error_keys = collect();

                foreach($files as $file_key => $file) {
                    $image_size = env('CLAIM_IMAGE_MAX_SIZE', config('upload.max_file_size'));
                    $image_type = env('CLAIM_IMAGE_TYPE', 'jpeg,png,jpg');

                    $validator = Validator::make($request->file(), [
                        $file_key => 'image|mimes:' . $image_type . '|max:' . $image_size
                    ]);

                    if($validator->fails()) {
                        $image_errors = array_merge($image_errors, json_decode(json_encode($validator->messages()), true));
                        $error_keys->push($file_key);
                    }
                }

                if ($image_errors) {
                    $image_errors = array_merge(['image' => trans('claims.api.errors.without_images')], $image_errors);
                }

                foreach($files as $file_key => $file) {
                    try {
                        if (!$error_keys->contains($file_key)) {
                            $image = new ClaimImage;
                            $image->addImage($file, $claim->id);
                            $image->save();
                            $claim->load('images');
                        }
                    } catch(\Exception $e) {
                        return response()->json(['message' => trans('claims.images_upload_fail')], 400);
                    }
                }

                if(!empty($image_errors)) {
                    $error = ValidationException::withMessages($image_errors);
                    throw $error;
                }
            } else {
                return response()->json(['message' => trans('claims.images_max_count_fail')], 400);
            }

        }

        $statuses = Claim::claimStatuses();
        $claim->type = $claim->getType($claim->type_id);
        $claim->status = $statuses[$claim->status];
        $claim->load('images');

        return $claim;
    }

    public function clearPhone($phone) {
        $phone = preg_replace('/^78/', '+7', $phone);
        $phone = str_replace('+7', '', $phone);
        $phone = str_replace(' ', '', $phone);
        $phone = str_replace('-', '', $phone);
        $phone = str_replace('(', '', $phone);
        $phone = str_replace(')', '', $phone);
        return $phone;
    }
}