<?php

namespace App\Http\Controllers\Api\Auth;

use App\Http\Controllers\Controller;
use App\Models\Apartment;
use App\Models\ApartmentSipNumber;
use App\Models\Building;
use App\Models\Device;
use App\Models\PermissionManager\Permission;
use App\Models\User;
use Backpack\Settings\app\Models\Setting;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;

class LoginController extends Controller
{

    use AuthenticatesUsers {
        login as traitLogin;
    }

    /**
     * Send the response after the user was authenticated.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Apartment  $apartment
     * @return \Illuminate\Http\JsonResponse | array
     */
    protected function sendLoginResponse(Request $request, Apartment $apartment = null)
    {
        $this->clearLoginAttempts($request);

        $user = $this->guard()->user();

        $personalAccessToken = $user->createToken(config('auth.oauth_client_name'));

        $user->withAccessToken($personalAccessToken->token);

        // each login should create a new Device
        $device = new Device;
        $device->id = $personalAccessToken->token->id;
        $device->locale = App::getLocale();

        $sip_client_number = null;
        $sip_client_password = null;
        $limit_of_activations = null;
        $number_of_activations = null;

        // get default sip values from Config
        list($sip_server_address, $sip_server_port, $concierge_sip_number) = array_values(Config::getMany([
            'settings.sip_server_address',
            'settings.sip_server_port',
            'settings.concierge_sip_number',
        ]));

        if ($apartment) {
            $device->apartment_id = $apartment->id;

            // this is building or house (aka object)
            $object = $apartment->object;

            // get sip configuration from object
            $sip_server_address   = $object->sip_server_address;
            $sip_server_port      = $object->sip_server_port;
            $concierge_sip_number = $object->concierge_sip_number;

            $busySipNumbers = $this->getBusySipNumbers($apartment);
            $allApartmentSipNumbers = $apartment->sip_numbers()->pluck('id')->toArray();

            $freeSipNumbers = [];
            foreach ($allApartmentSipNumbers as $id) {
                if (!in_array($id,$busySipNumbers)) {
                    $freeSipNumbers[] = $id;
                }
            }

            $sip_client_number = null;
            $sip_client_password = null;
            $sipNumberId = array_pop($freeSipNumbers);

            if ($sipNumberId) {
                // Set sip number and passwords
                $apartment_sip_number = ApartmentSipNumber::find($sipNumberId);
                if ($apartment_sip_number instanceof ApartmentSipNumber) {
                    $sip_client_number = $apartment_sip_number->sip_number;
                    $sip_client_password = $apartment_sip_number->sip_password ? (string)$apartment_sip_number->sip_password : null;
                }
                $device->sip_number_id = $sipNumberId;
            }

            $limit_of_activations = $apartment->activation_limit;
            $number_of_activations = $apartment->getActivationsUsed();
        }

        $device->save();

        return array_merge($user->toArray(), [
            'api_token' => $personalAccessToken->accessToken,
            'sip_server_address' => $sip_server_address,
            'sip_server_port' => $sip_server_port,
            'concierge_sip_number' => $concierge_sip_number,
            'sip_client_number' => $sip_client_number,
            'sip_client_password' => $sip_client_password,
            'limit_of_activations' => $limit_of_activations,
            'number_of_activations' => $number_of_activations
        ]);
    }

    private function getBusySipNumbers(Apartment $apartment)
    {
        $busySipNumbers = [];
        $users = $apartment->users()->withCount('tokens')->get();
        foreach ($users as $user) {
            $numbers = $user->devices()->whereNotNull('sip_number_id')->pluck('sip_number_id')->toArray();
            $busySipNumbers = array_merge($busySipNumbers, $numbers);
        }
        return $busySipNumbers;
    }

    /**
     * Стандартный логин по email и паролю не предусмотрен
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\Response
     */
    public function login(Request $request)
    {
        if (\App::environment('production')) {
            abort(404);
        } else {
            return $this->traitLogin($request);
        }

    }

    /**
     * Delete token user has authorized with
     */
    public function logout()
    {
        // delete token and related device (by database constraint ON CASCADE)
        Auth::user()->token()->delete();
    }

    /**
     * Авторизация пользователей СКА
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function acsAuthorization(Request $request)
    {
        // Поиск пользователя в бд
        $count = User::where('email', $request->get('email'))->get()->count();

        // Получить значение настройки, позволяющей создавать администраторов
        $setting = DB::table('settings')->where('key', 'acs_users')->first();

        if ($setting) {
            $administators = (bool)$setting->value;
        } else {
            $administators = false;
        }

        if (!$count && $administators) {
            // Внести нового пользователя в бд
            $user = User::create([
                'name' => $request->get('name'),
                'email' => $request->get('email'),
                'password' => $request->get('password')
            ]);

            // Выдать ему права на все объекты
            // Найти id права в бд на полный доступ
            $permission_id = Permission::where('name', Permission::FULL_ACCESS)->get()->first()->id;

            foreach (Building::all('id') as $building) {
                DB::table('permission_users')->insert([
                    'user_id' => $user->id,
                    'permission_id' => $permission_id,
                    'object_id' => $building->id
                ]);
            }
        }

        // Вернуть в ответе количество найденных пользователей
        return response()->json(['count' => $count, 'administrators' => $administators]);
    }
}
