<?php

namespace App\Http\Controllers\Admin;

use App\Models\PermissionManager\Permission;
use App\Models\PermissionManager\PermissionGroup;
use App\Models\User;
use App\Rules\Users\HaveAccessToEntry;
use App\Rules\Users\IsActive;
use App\Traits\Admin\Ordering;
use App\Traits\PermissionManager\GroupPermission;
use App\Traits\PermissionManager\CheckingPermissions;
use Illuminate\Http\Request as CrudRequest;
use Backpack\CRUD\app\Http\Controllers\CrudController;
use App\Http\Requests\UserStoreCrudRequest as StoreRequest;
use App\Http\Requests\UserUpdateCrudRequest as UpdateRequest;

use Backpack\CRUD\Exception\AccessDeniedException;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

/**
 * Class UserCrudController
 *
 * @package App\Http\Controllers\Admin
 */
class UserCrudController extends CrudController
{
    use CheckingPermissions, Ordering;

    /**
     * @var string $permissionsRelation
     */
    protected $permissionsRelation = 'permissions';

    /**
     * @var string $rolesRelation
     */
    protected $rolesRelation = 'roles';

    /**
     * UserCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->applyCheckingPermissions([[
            [Permission::USER_CREATE_ACCESS],
            ['create', 'store'],
        ], [
            [Permission::USER_VIEW_FULL_ACCESS, Permission::USER_VIEW_LIMITED_ACCESS],
            ['index'],
        ], [
            [Permission::USER_EDIT_FULL_ACCESS, Permission::USER_EDIT_ACCESS],
            ['edit', 'update'],
        ], [
            [Permission::USER_DELETE_ACCESS],
            ['destroy']
        ]]);
    }


    /**
     * Allow developers to set their configuration options for a CrudPanel.
     */
    public function setup()
    {
        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel(config('backpack.permissionmanager.user_model'));
        $this->crud->setEntityNameStrings(trans('backpack::permissionmanager.user'), trans('backpack::permissionmanager.users'));
        $this->crud->setRoute(config('backpack.base.route_prefix').'/user');

        $this->setButtons();

        $this->applyBackpackPermissions();

        // Columns.
        $this->crud->setColumns([
            [
                'name'  => 'name',
                'label' => trans('backpack::permissionmanager.name'),
                'type'  => 'text',
            ],
            [
                'name'  => 'email',
                'label' => trans('backpack::permissionmanager.email'),
                'type'  => 'email',
            ],
            [ // n-n relationship (with pivot table)
                'label'     => trans('backpack::permissionmanager.roles'), // Table column heading
                'type'      => 'select_multiple',
                'name'      => 'roles', // the method that defines the relationship in your Model
                'entity'    => 'roles', // the method that defines the relationship in your Model
                'attribute' => 'name', // foreign key attribute that is shown to user
                'model'     => config('laravel-permission.models.role'), // foreign key model
            ],
            [ // n-n relationship (with pivot table)
                'label'     => trans('backpack::permissionmanager.extra_permissions'), // Table column heading
                'type'      => 'select_multiple',
                'name'      => 'permissions', // the method that defines the relationship in your Model
                'entity'    => 'permissions', // the method that defines the relationship in your Model
                'attribute' => 'translated_name', // foreign key attribute that is shown to user
                'model'     => config('laravel-permission.models.permission'), // foreign key model
            ],
        ]);

        if (Auth::user()->can(Permission::USER_EDIT_FULL_ACCESS)) {
            $this->crud->addField([
                'name' => 'is_active',
                'label' => __('permissionmanager.is_active'),
                'type' => 'checkbox',
                'hint' => __('permissionmanager.is_active_hint')
            ], 'update');
        }

        // Fields
        $this->crud->addFields([
            [
                'name'  => 'name',
                'label' => trans('backpack::permissionmanager.name'),
                'type'  => 'text',
            ],
            [
                'name'  => 'email',
                'label' => trans('backpack::permissionmanager.email'),
                'type'  => 'text',
            ],
            [
                'name'  => 'password',
                'label' => trans('backpack::permissionmanager.password'),
                'type'  => 'password',
            ],
            [
                'name'  => 'password_confirmation',
                'label' => trans('backpack::permissionmanager.password_confirmation'),
                'type'  => 'password',
            ],
            [
                // two interconnected entities
                'label'             => trans('backpack::permissionmanager.user_role_permission'),
                'field_unique_name' => 'user_role_permission',
                'type'              => 'permissions_checklist_dependency',
                'name'              => 'roles_and_permissions', // the methods that defines the relationship in your Model
                'subfields'         => [
                    'primary' => [
                        'label'            => trans('backpack::permissionmanager.roles'),
                        'name'             => 'roles', // the method that defines the relationship in your Model
                        'entity'           => 'roles', // the method that defines the relationship in your Model
                        'entity_secondary' => 'permissions', // the method that defines the relationship in your Model
                        'attribute'        => 'name', // foreign key attribute that is shown to user
                        'model'            => config('laravel-permission.models.role'), // foreign key model
                        'pivot'            => true, // on create&update, do you need to add/delete pivot table entries?]
                        'number_columns'   => 3, //can be 1,2,3,4,6
                    ],
                    'secondary' => [
                        'label'          => ucfirst(trans('backpack::permissionmanager.permission_singular')),
                        'name'           => 'permissions', // the method that defines the relationship in your Model
                        'entity'         => 'permissions', // the method that defines the relationship in your Model
                        'entity_primary' => 'roles', // the method that defines the relationship in your Model
                        'attribute'      => 'translated_name', // foreign key attribute that is shown to user
                        'model'          => Permission::class, // foreign key model
                        'pivot'          => true, // on create&update, do you need to add/delete pivot table entries?]
                        'number_columns' => 3, //can be 1,2,3,4,6
                    ],
                ],
            ],
        ]);

        if (Auth::user()->can(Permission::CLAIMS_BY_TYPE_ACCESS)) {
            $this->crud->addFields([[   // SelectMultiple = n-n relationship (with pivot table)
                'label' => __('claims.claims_types'),
                'permission' => 'Заявки: Заявки по типу',
                'type' => 'select2_multiple',
                'name' => 'claimTypes', // the method that defines the relationship in your Model
                'entity' => 'claimTypes', // the method that defines the relationship in your Model
                'attribute' => 'text', // foreign key attribute that is shown to user
                'model' => "App\Models\ClaimType", // foreign key model
                'pivot' => true, // on create&update, do you need to add/delete pivot table entries?
            ]]);
        }

        $this->crud->addColumn([
            'name' => 'is_active',
            'label' => __('permissionmanager.is_active'),
            'type' => 'boolean',
        ])->afterColumn('email');

        $this->crud->setEntityNameStrings(__('permissionmanager.user_singular'), __('permissionmanager.user_plural'));

        // should display here only tenants
        $users = User::with('apartments')->get();
        $users_without_apartments = [];

        foreach ($users as $user) {
            if($user->apartments()->count() == 0) $users_without_apartments[] = $user->id;
        }

        $this->crud->addClause('whereIn', 'id', $users_without_apartments);

        $this->crud->setListView('crud::user.list');
        $this->crud->setCreateView('crud::user.create');
        $this->crud->setEditView('crud::user.edit');
    }

    /**
     * Store a newly created resource in the database.
     *
     * @param \App\Http\Requests\UserStoreCrudRequest $request
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(CrudRequest $request)
    {
        // Подготовить текущий запрос к валидации, внести в него корректные данные
        $request = $this->setRequest($request);

        // Проверить запрос на соответствие шаблону (валидация входных данных, включая подготовленные роли)
        $this->checkRequest($request, 'create');

        $this->handlePasswordInput($request);

        return parent::storeCrud(
            $this->prepareRolesByObjectInRequest(
                $this->preparePermissionsByObjectInRequest($request)
            )
        );
    }

    /**
     * Update the specified resource in the database.
     *
     * @param \App\Http\Requests\UserUpdateCrudRequest $request
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(CrudRequest $request)
    {
        $user = User::find($request->get('id'));
        if(!$user){
            abort(404);
        }
        if (Auth::user()->getPermissionsWeight() < $user->getPermissionsWeight()) {
            abort(403);
        }

        // Если пользователь пытается редактировать сам себя, то изменение прав не нужно
        if (Auth::user()->id === (int)$this->request->get('id')) {
            // Очистить права и роли, которые пришли
            $request->request->set('permissions', collect());
            $request->request->set('roles', collect());

            $request = $this->addOldRolesInToRequest($request, true);
            $request = $this->addOldPermissionsInToRequest($request, true);
        } else {
            // Подготовить текущий запрос к валидации, внести в него корректные данные
            $request = $this->setRequest($request);;
            // Проверить запрос на соответствие шаблону (валидация входных данных, включая подготовленные роли)
            $this->checkRequest($request, 'update');

            if (!Auth::user()->can(Permission::FULL_ACCESS)) {
                // Добавить старые права в результирующий запрос
                $request = $this->addOldPermissionsInToRequest($request);

                // Добавить старые роли в результирующий запрос
                $request = $this->addOldRolesInToRequest($request);
            }
        }

        $this->handlePasswordInput($request);

        return parent::updateCrud(
            $this->prepareRolesByObjectInRequest(
                $this->preparePermissionsByObjectInRequest($request)
            )
        );
    }

    /**
     * Выполнить коррекцию входящего запроса в соответствии с группами прав и установленными ролями.
     *
     * @return \Backpack\CRUD\app\Http\Requests\CrudRequest
     */
    protected function setRequest(CrudRequest $request)
    {
        $request = $this->setPermissionsByRoles($request);
        $request = $this->setGroupPermissions($request);

        // Преобразовать массив ролей в коллекцию и id ролей в тип int
        $request->request->add(['roles' => collect($request->input('roles'))->transform(function ($role) {
            return (int)$role;
        })]);

        // Преобразовать массив прав в коллекцию и id ролей в тип int
        $request->request->add(['permissions' => collect($request->input('permissions'))->transform(function ($permission) {
            return (int)$permission;
        })]);

        return $request;
    }

    /**
     * Установить все права, согласно ролям во входящем запросе
     *
     * @return \Backpack\CRUD\app\Http\Requests\CrudRequest
     */
    private function setPermissionsByRoles(CrudRequest $request)
    {
        $roles = collect($request->roles);
        $permissions = collect($request->permissions);
        $permissions->transform(function ($permission) {
            return $permission = (int)$permission;
        });

        $role_permissions = $full_permissions = collect();

        $roles->each(function ($role) use (&$role_permissions) {
            $role_permissions = $role_permissions->merge(DB::table('permission_roles')->select('permission_id')->where('role_id', $role)->pluck('permission_id'));
        });

        $full_permissions = $role_permissions->merge($permissions)->unique()->values();

        $request->request->add(['permissions' => $full_permissions->toArray()]);

        return $request;
    }

    /**
     * Устанавливает права из одной группы, обладающие меньшим весом
     *
     * @return \Backpack\CRUD\app\Http\Requests\CrudRequest
     */
    private function setGroupPermissions(CrudRequest $request)
    {
        $input_permissions = collect($request->input('permissions'));

        // Проверить правила на наличие полного доступа
        if (($full_access_request = $this->checkFullAccessPermission($request)) instanceof CrudRequest) {
            return $full_access_request;
        }

        // Получить все веса входящих прав из базы данных
        $input_permissions_info = DB::table('permissions')
            ->select('id', 'weight', 'group_id')
            ->whereIn('id', $input_permissions)
            ->get();

        // Найти все права, ниже текущих по весам
        $raw_permissions = collect();
        $input_permissions_info->each(function ($permission) use (&$raw_permissions) {
            $raw_permissions = $raw_permissions->merge(DB::table('permissions')
                ->select('id')
                ->where('weight', '<', $permission->weight)
                ->where('group_id', $permission->group_id)
                ->get()->pluck('id'));
        });

        // Сформировать массив прав, удалив все повторяющиеся
        $permissions = $raw_permissions->merge($input_permissions)->unique();

        // Внести в запрос обновленные данные
        $request->request->add(['permissions' => $permissions]);

        return $request;
    }

    /**
     * Проверка правил на наличие полного доступа
     *
     * @param CrudRequest $request
     * @return CrudRequest|null
     */
    protected function checkFullAccessPermission(CrudRequest $request)
    {
        $input_permissions = collect($request->input('permissions'));

        // Проверить на наличие права "полный доступ" в случае его наличия удалить остальные права
        $full_access = Permission::where('name', Permission::FULL_ACCESS)->first();

        // Если есть право "полный доступ", вернуть только его
        if ($input_permissions->contains($full_access->id)) {
            $request->request->add(['permissions' => collect($full_access->id)]);
            return $request;
        }

        return null;
    }

    /**
     * Добавлять старые права к новым правам
     * Работает только для тех пользователей, которым не доступен полный список прав
     *
     * @return \Backpack\CRUD\app\Http\Requests\CrudRequest
     */
    protected function addOldPermissionsInToRequest(CrudRequest $request, $owner = false)
    {
        // Получить старые права
        $old_permissions = $this->getRulesFromDb('permission_users', 'permission_id', (int)$request->input('id'));
        // Получить права пользователя
        $user_permissions = $this->getRulesFromDb('permission_users', 'permission_id', (int)Auth::user()->id);

        // Определить права, которые будут добавлены в результирующий запрос
        if ($owner) {
            $request->request->add(['permissions' => $old_permissions]);
        } else {

            $request->request->add(['permissions' => $request->input('permissions')->merge($old_permissions->diff($user_permissions))]);

            $full_access = Permission::where('name', Permission::FULL_ACCESS)->first();

            // Если есть право "полный доступ", вернуть только его
            if ($request->input('permissions')->contains($full_access->id)) {
                $request->request->add(['permissions' => collect($full_access->id)]);
                return $request;
            }
        }

        return $request;
    }

    /**
     * Добавить старые роли к новым ролям
     *
     * @return \Backpack\CRUD\app\Http\Requests\CrudRequest
     */
    protected function addOldRolesInToRequest(CrudRequest $request, $owner = false)
    {
        // Получить старые роли
        $old_roles = $this->getRulesFromDb('role_users', 'role_id', (int)$request->input('id'));
        // Получить роли пользователя
        $user_roles = $this->getRulesFromDb('role_users', 'role_id', (int)Auth::user()->id);

        if ($owner) {
            $request->request->add(['roles' => $old_roles]);
        } else {
            // Определить роли, которые будут добавлены в результирующий запрос
            $roles = $request->input('roles')->merge($old_roles->diff($user_roles));

            $request->request->add(['roles' => $roles]);
        }


        return $request;
    }

    /**
     * Получает требуемый список доступных правил из базы данных
     *
     * @param string $table
     * @param int $rule_id
     * @param int $user_id
     *
     * @return \Illuminate\Support\Collection
     */
    private function getRulesFromDb(string $table, string $rule, int $user_id)
    {
        return DB::table($table)
            ->select($rule)
            ->where('object_id', config('app.object.id'))
            ->where('user_id', $user_id)
            ->pluck($rule);
    }

    /**
     * Проверка запроса на соответствие шаблону
     *
     */
    protected function checkRequest(CrudRequest $request, $action = 'create')
    {
        $user_id = (int)$request->id;

        // Подготовка правил валидации
        $roles = $permissions = [];

        if (!Auth::user()->can(Permission::FULL_ACCESS)) {
            $roles          = [new HaveAccessToEntry('role_users', 'role_id', 'not_possible_role')];
            $permissions    = [new HaveAccessToEntry('permission_users', 'permission_id', 'not_possible_permission')];
        }

        switch ($action)
        {
            case 'update':
                $rules = [
                    'email' => 'required|email',
                    'name' => 'required',
                    'password' => 'confirmed',
                    'roles' => $roles,
                    'permissions' => $permissions,
                    'is_active' => [new IsActive($user_id)]
                ];
                break;

            case 'create':
                $rules = [
                    'email'    => 'required|unique:'.config('laravel-permission.table_names.users', 'users').',email|email',
                    'name'     => 'required',
                    'password' => 'required|confirmed',
                    'roles' => $roles,
                    'permissions' => $permissions,
                    'is_active' => [new IsActive($user_id)]
                ];
                break;
            default:
                $rules = [];
                break;
        }

        Validator::validate($request->all(), $rules);
    }

    /**
     *
     * Проверить входящие роли на допустимость
     *
     * @return boolean
     */
    protected function checkRoles(Collection $roles = null)
    {
        // Если входищих ролей нет, то и проверять нечего
        if (empty($roles)) {
            return true;
        }

        return false;
    }

    /**
     * @param \Backpack\CRUD\app\Http\Requests\CrudRequest $request
     *
     * @return \Backpack\CRUD\app\Http\Requests\CrudRequest
     */
    private function preparePermissionsByObjectInRequest(CrudRequest $request)
    {
        if ($permissions = $request->input($this->permissionsRelation)) {
            $permissions = collect($permissions)->mapWithKeys(function ($permissionId) {
                return [
                    $permissionId => [
                        'object_id' => config('app.object.id'),
                    ]
                ];
            })->toArray();

            $request->request->set($this->permissionsRelation, $permissions);
        }

        return $request;
    }

    /**
     * @param \Backpack\CRUD\app\Http\Requests\CrudRequest $request
     *
     * @return \Backpack\CRUD\app\Http\Requests\CrudRequest
     */
    private function prepareRolesByObjectInRequest(CrudRequest $request)
    {
        if ($roles = $request->input($this->rolesRelation)) {
            $roles = collect($roles)->mapWithKeys(function ($roleId) {
                return [
                    $roleId => [
                        'object_id' => config('app.object.id'),
                    ]
                ];
            })->toArray();

            $request->request->set($this->rolesRelation, $roles);
        }

        return $request;
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     *
     * @return Response
     */
    public function edit($id)
    {
        $user = User::find($id);
        if(!$user) {
            abort(404);
        }
        if (Auth::user()->getPermissionsWeight() < User::find($id)->getPermissionsWeight()) {
            abort(403);
        }

        $this->crud->hasAccessOrFail('update');

        if (Auth::user()->getAuthIdentifier() === (int)$id) {
            $this->crud->removeField('is_active');
        }

        // get the info for that entry
        $this->data['entry'] = $this->crud->getEntry($id);
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['title'] = trans('backpack::crud.edit').' '.$this->crud->entity_name;

        $this->data['id'] = $id;

        // Вернуть все права и роли из бд
        $this->data['roles'] = $this->getRoles()->toJson();
        $this->data['permissions'] = $this->getPermissions()->toJson();

        // Является ли данная страница страницей пользователя
        if (Auth::user()->id === (int)$id) {
            $this->data['owner'] = 'true';
        } else {
            $this->data['owner'] = 'false';
        }

        $this->addFullAccessData($id);

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getEditView(), $this->data);
    }

    protected function addFullAccessData($id)
    {
        $full_access = Permission::where('name', Permission::FULL_ACCESS)->first();

        $count = DB::table('permission_users')
            ->where('user_id', $id)
            ->where('object_id', config('app.object.id'))
            ->where('permission_id', $full_access->id)
            ->count();

        $this->data['full_access'] = json_encode($count? true : false);
    }

    /**
     * Получить список ролей и принадлежащих им прав в виде коллекции
     *
     * @return \Illuminate\Support\Collection
     */
    private function getRoles()
    {
        $raw_roles = DB::table('permission_roles')->get();

        $roles = collect();
        $raw_roles->each(function ($role) use (&$roles) {
            if ($roles->has($role->role_id)) {
                $roles->put($role->role_id, $roles->get($role->role_id)->merge($role->permission_id));
            } else {
                $roles->put($role->role_id, collect($role->permission_id));
            }
        });

        return $roles;
    }

    /**
     * Получить список прав в виде коллекции
     *
     */
    private function getPermissions()
    {
        $raw_permissions = DB::table('permissions')
            ->select('id', 'group_id AS group', 'weight', 'name')
            ->get();

        $permissions = collect();
        $raw_permissions->each(function ($permission) use (&$permissions) {
            $permission_data = collect(['group' => $permission->group, 'weight' => $permission->weight]);

            if ($permission->name === Permission::FULL_ACCESS) {
                $permission_data->put('full_access', true);
            }

            $permissions->put($permission->id, $permission_data);
        });

        return $permissions;
    }

    /**
     * Show the form for creating inserting a new row.
     *
     * @return Response
     */
    public function create()
    {
        $this->crud->hasAccessOrFail('create');

        // prepare the fields you need to show
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getCreateFields();
        $this->data['title'] = trans('backpack::crud.add').' '.$this->crud->entity_name;

        $this->data['roles'] = $this->getRoles()->toJson();
        $this->data['permissions'] = $this->getPermissions()->toJson();

        $this->data['full_access'] = json_encode(false);

        $this->data['owner'] = 'false';

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getCreateView(), $this->data);
    }

    /**
     * Переопределение необходимо для сортировки ненастраиваемых полей
     *
     * @return array|\Backpack\CRUD\app\Http\Controllers\Operations\JSON
     * @throws \Backpack\CRUD\Exception\AccessDeniedException
     */
    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }

        // structure the response in a DataTable-friendly way
        if (Auth::user()->cannot(Permission::USER_VIEW_FULL_ACCESS)) {
            $subordinateUsersByPermissions = Auth::user()->getSubordinateByPermissions()->pluck('id')->push(Auth::user()->id);
            $this->crud->addClause('whereIn', 'id', $subordinateUsersByPermissions->toArray());
        }

        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);

            if(in_array($column['name'], ['name'])) {
                // для поля с именем number установлена отдельная сортировка для более удобочитаемого отображения квартир, имеющих номера
                $this->crud->query = $this->crud->query->orderByRaw(
                    $column['name'] . '+0 '.$column_direction.', '.
                    $column['name'].' '.$column_direction.
                    ', LENGTH('.$column['name'].') '.$column_direction
                );
            } elseif ($column['tableColumn']) {
                // clear any past orderBy rules
                $this->crud->query->getQuery()->orders = null;
                // apply the current orderBy rules
                $this->crud->orderBy($column['name'], $column_direction);
            }
        }

        // Если сортировка кастомная, то присваиваем отсортированные записи
        $entries = $this->crud->getEntries();

        return $this->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);
    }

    /**
     * Handle password input fields.
     *
     * @param CrudRequest $request
     */
    protected function handlePasswordInput(CrudRequest $request)
    {
        // Remove fields not present on the user.
        $request->request->remove('password_confirmation');

        // Encrypt password if specified.
        if ($request->input('password')) {
            $request->request->set('password', bcrypt($request->input('password')));
        } else {
            $request->request->remove('password');
        }
    }

    protected function setButtons()
    {
        $this->crud->removeAllButtons();

        if (Auth::user()->can(Permission::USER_CREATE_ACCESS)) {
            $this->crud->addButton('top', 'create', 'view', 'crud::buttons.create');
        }
        if (Auth::user()->can(Permission::USER_DELETE_ACCESS)) {
            $this->crud->addButton('line', 'delete', 'view', 'vendor.backpack.crud.user.buttons.delete');
        }
        if (Auth::user()->can(Permission::USER_EDIT_FULL_ACCESS) || Auth::user()->can(Permission::USER_EDIT_ACCESS)) {
            $this->crud->addButton('line', 'update', 'view', 'crud::buttons.update');
        }
    }


    /**
     * Get the save configured save action or the one stored in a session variable.
     * @return [type] [description]
     */
    public function getSaveAction()
    {
        $saveAction = session('save_action', config('backpack.crud.default_save_action', 'save_and_back'));
        $saveOptions = [];
        $saveCurrent = [
            'value' => $saveAction,
            'label' => $this->getSaveActionButtonName($saveAction),
        ];

        switch ($saveAction) {
            case 'save_and_edit':
                $saveOptions['save_and_back'] = $this->getSaveActionButtonName('save_and_back');
                if (Auth::user()->can(Permission::USER_CREATE_ACCESS)) {
                    $saveOptions['save_and_new'] = $this->getSaveActionButtonName('save_and_new');
                }
                break;
            case 'save_and_new':
                if (Auth::user()->can(Permission::USER_CREATE_ACCESS)) {
                    $saveOptions['save_and_back'] = $this->getSaveActionButtonName('save_and_back');
                }
                if (Auth::user()->can(Permission::USER_EDIT_ACCESS) || Auth::user()->can(Permission::USER_EDIT_FULL_ACCESS)) {
                    $saveOptions['save_and_edit'] = $this->getSaveActionButtonName('save_and_edit');
                }
                break;
            case 'save_and_back':
            default:
                if (Auth::user()->can(Permission::USER_EDIT_ACCESS) || Auth::user()->can(Permission::USER_EDIT_FULL_ACCESS)) {
                    $saveOptions['save_and_edit'] = $this->getSaveActionButtonName('save_and_edit');
                }
                if (Auth::user()->can(Permission::USER_CREATE_ACCESS)) {
                    $saveOptions['save_and_new'] = $this->getSaveActionButtonName('save_and_new');
                }
                break;
        }

        return [
            'active' => $saveCurrent,
            'options' => $saveOptions,
        ];
    }

    /**
     * Get the translated text for the Save button.
     * @param  string $actionValue [description]
     * @return [type]              [description]
     */
    private function getSaveActionButtonName($actionValue = 'save_and_back')
    {
        switch ($actionValue) {
            case 'save_and_edit':
                return trans('backpack::crud.save_action_save_and_edit');
                break;
            case 'save_and_new':
                return trans('backpack::crud.save_action_save_and_new');
                break;
            case 'save_and_back':
            default:
                return trans('backpack::crud.save_action_save_and_back');
                break;
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     *
     * @return string
     */
    public function destroy($id)
    {
        if (Auth::user()->getPermissionsWeight() < User::find($id)->getPermissionsWeight()) {
            abort(403);
        }

        return parent::destroy($id);
    }

    public function show($id){
        Abort(404);
    }

    /**
     * Get the HTML of the cells in a table row, for a certain DB entry.
     * @param  Entity $entry A db entry of the current entity;
     * @param  int The number shown to the user as row number (index);
     * @return array         Array of HTML cell contents.
     */
    public function getRowViews($entry, $rowNumber = false)
    {
        $row_items = [];

        foreach ($this->crud->columns as $key => $column) {
            $row_items[] = $this->crud->getCellView($column, $entry, $rowNumber);
        }

        $buttons = collect();
        // Проверка на вес прав текущего пользователя относительно выводимого
        if (Auth::user()->getPermissionsWeight() >= User::find($entry->id)->getPermissionsWeight())
        {
            foreach ($this->crud->buttons as $button) {
                if ($button->name === 'update' || $button->name === 'delete') {
                    $buttons->push($button);
                }
            }
        }

        // add the buttons as the last column
        if ($buttons->count()) {
            $row_items[] = \View::make('crud::inc.button_stack_custom', ['stack' => 'line'])
                ->with('crud', $this->crud)
                ->with('entry', $entry)
                ->with('buttons', $buttons)
                ->render();
        } else {
            $row_items[] = '<td></td>';
        }

        // add the details_row button to the first column
        if ($this->crud->details_row) {
            $details_row_button = \View::make('crud::columns.details_row_button')
                ->with('crud', $this)
                ->with('entry', $entry)
                ->with('row_number', $rowNumber)
                ->render();
            $row_items[0] = $details_row_button.$row_items[0];
        }

        return $row_items;
    }

    /**
     * Created the array to be fed to the data table.
     *
     * @param $entries Eloquent results.
     * @return array
     */
    public function getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex = false)
    {
        $rows = [];

        foreach ($entries as $row) {
            $rows[] = $this->getRowViews($row, $startIndex === false ? false : ++$startIndex);
        }

        return [
            'draw'            => (isset($this->request['draw']) ? (int) $this->request['draw'] : 0),
            'recordsTotal'    => $totalRows,
            'recordsFiltered' => $filteredRows,
            'data'            => $rows,
        ];
    }
}