<?php

namespace App\Http\Controllers\Admin;

use App\Models\PermissionManager\Permission;
use App\Models\Quiz;
use App\Models\User;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\CRUD\app\Http\Requests\CrudRequest;
use Illuminate\Support\Facades\Auth;
use App\Models\QuizAnswer;
use App\Models\QuizDestination;
use App\Models\QuizTemplate;
use App\Models\Category;
use App\Models\QuizResult;
use Backpack\CRUD\app\Http\Controllers\CrudController;
use App\Http\Requests\QuizRequest as StoreRequest;
use App\Http\Requests\QuizRequest as UpdateRequest;
use App\Models\Apartment;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Validator;

class QuizCrudController extends CrudController
{
    use CheckingPermissions;

    /**
     * QuizCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::QUIZES_FULL_ACCESS],
            [
                'create',
                'store',
                'index',
                'edit',
                'update',
                'destroy',
                'results',
                'printResults',
            ],
        ]]);
    }

    public function setup()
    {

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\Quiz');
        $this->crud->setRoute(config('backpack.base.route_prefix') . '/quizzes');
        $this->crud->setEntityNameStrings(__('quizzes.quiz_singular'), __('quizzes.quiz_plural'));

        $this->applyBackpackPermissions();

        $this->crud->setCreateView('crud::quiz.create');
        $this->crud->setEditView('crud::quiz.edit');
        $this->crud->setShowView('crud::quiz.show');

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */

        $this->crud->addClause('where', 'object_id', '=', config('app.object.id'));

        $this->crud->addColumn([
            'name' => 'whom',
            'label' => __('quizzes.crud_fields.whom'),
            'type' => 'model_function',
            'function_name' => 'getWhom'
        ]);
        $this->crud->addColumn([
            'name' => 'title',
            'label' => __('quizzes.crud_fields.title'),
            'type' => 'text'
        ]);

        $this->crud->addColumn([
            'name' => 'status',
            'label' => __('quizzes.crud_fields.status'),
            'type' => 'text',
        ]);

        $this->crud->addColumn([
            'name' => 'is_private',
            'label' => __('quizzes.crud_fields.is_private'),
            'type' => 'boolean',
        ]);
        $this->crud->addColumn([
            'name' => 'created_at',
            'label' => __('quizzes.crud_fields.created_at'),
            'type' => 'text_datetime_utc',
        ]);
        $this->crud->addColumn([
            'name' => 'start_at',
            'label' => __('quizzes.crud_fields.start_at'),
            'type' => 'text',
        ]);
        $this->crud->addColumn([
            'name' => 'end_at',
            'label' => __('quizzes.crud_fields.end_at'),
            'type' => 'text',
        ]);

        // fields for forms

        $this->crud->addField([
            'name' => 'destinations',
            'label' => __('quizzes.form_fields.destination_edit'),
            'type' => 'hidden',
            'attributes' => [
                'id' => 'destinations-field',
            ],
        ]);

        $apartment = new Apartment;

        $this->crud->addField([
            'name' => 'object_id',
            'type' => 'hidden',
            'value' => config('app.object.id')
        ]);

        $this->crud->addField([
            'name' => 'destinations-list',
            'type' => 'custom_html',
            'is_destination' => TRUE,
            'value' => '<label>' . __('quizzes.form_fields.destination') . '</label><div class="destinations-list"></div>',
        ]);

        $this->crud->addField([
            'name' => 'is_all_house',
            'label' => __('quizzes.form_fields.is_all_house'),
            'type' => 'checkbox',
            'col' => 'col1',
            'is_destination' => TRUE,
            'attributes' => [
                'id' => 'is_all_house-field',
            ],
        ], 'create');

        $this->crud->addField([
            'name' => 'entrance',
            'label' => __('quizzes.form_fields.select_entrance'),
            'type' => 'select2_from_array',
            'options' => $apartment->getEntrances(),
            'allows_null' => TRUE,
            'col' => 'col1',
            'is_destination' => TRUE,
            'attributes' => [
                'id' => 'entrance-field',
                'data-disable' => 'is_all_house',
            ],
        ]);

        $this->crud->addField([
            'name' => 'floor',
            'label' => __('quizzes.form_fields.select_floor'),
            'type' => 'select2_from_array',
            'options' => $apartment->getFloors(),
            'allows_null' => TRUE,
            'col' => 'col1',
            'is_destination' => TRUE,
            'attributes' => [
                'id' => 'floor-field',
                'data-disable' => 'is_all_house',
            ],
        ]);


        $this->crud->addField([
            'name' => 'add-destination-group',
            'col' => 'col1',
            'is_destination' => TRUE,
            'type' => 'custom_html',
            'value' => '<button id="add-destination-group" class="btn btn-primary" data-disable="is_all_house">' . __('destinations.add_destination_group') . '</button>',
        ], 'create');

        $this->crud->addField([
            'name' => 'apartment',
            'label' => __('quizzes.form_fields.select_apartment'),
            'type' => 'select2_from_array',
            'col' => 'col2',
            'is_destination' => TRUE,
            'options' => $apartment->getFlats(),
            'allows_null' => TRUE,
            'attributes' => [
                'id' => 'apartment-field',
                'data-disable' => 'is_all_house',
            ],
        ]);

        $this->crud->addField([
            'name' => 'add-destination',
            'type' => 'custom_html',
            'col' => 'col2',
            'is_destination' => TRUE,
            'value' => '<button id="add-destination" class="btn btn-primary" data-disable="is_all_house">' . __('destinations.add_destination') . '</button>',
        ], 'create');

        $this->crud->addField([
            'name' => 'title',
            'label' => __('quizzes.crud_fields.title'),
            'type' => 'text',
            'attributes' => [
                'id' => 'title'
            ]
        ]);

        $this->crud->addField([
            'name' => 'event_date_range', // a unique name for this field
            'start_name' => 'start_at', // the db column that holds the start_date
            'end_name' => 'end_at', // the db column that holds the end_date
            'label' => __('quizzes.crud_fields.date_range'),
            'type' => 'date_range',
            // OPTIONALS
            'start_default' => date('Y-m-d H:i:s', strtotime('+3 hours')), // default value for start_date
            'end_default' => date('Y-m-d H:i:s', strtotime('+1 month +3 hours')), // default value for end_date
            'date_range_options' => [ // options sent to daterangepicker.js
                'timePicker' => true,
                'locale' => ['format' => 'YYYY-MM-DD HH:mm:ss']
            ]

        ], 'create');

        $this->crud->addField([
            'name' => 'start_at',
            'label' => __('quizzes.crud_fields.start_at'),
            'type' => 'text',
            'attributes' => [
                'id' => 'start_at-field',
                'readonly' => 'true',
            ],
        ], 'update');

        $this->crud->addField([
            'name' => 'end_at',
            'label' => __('quizzes.crud_fields.end_at'),
            'type' => 'text',
            'attributes' => [
                'id' => 'end_at-field',
                'readonly' => 'true',
            ],
        ], 'update');

        $this->crud->addField([
            'name' => 'template',
            'value' => '<button class="btn btn-primary" type="button" data-toggle="modal" data-target="#templates-picker">'
                . __('quizzes.buttons.template') . '</button>',
            'type' => 'custom_html',
        ]);

        $this->crud->addField([
            'name' => 'question',
            'label' => __('quizzes.crud_fields.question'),
            'type' => 'textarea',
            'attributes' => [
                'id' => 'question'
            ]
        ]);

        $this->crud->addField([
            'name' => 'answers',
            'label' => __('quizzes.crud_fields.variants'),
            'type' => 'table',
            'columns' => ['text' => ''],
            'entity_singular' => __('quizzes.crud_fields.variant_add_button'), // used on the "Add X" button
            'max' => 20, // maximum rows allowed in the table
            'min' => 0 // minimum rows allowed in the table
        ]);

        $this->crud->addField([
            'name' => 'free_answer',
            'label' => __('quizzes.crud_fields.free_answer'),
            'type' => 'checkbox',
            'attributes' => [
                'id' => 'free_answer'
            ]
        ]);

        $this->crud->addField([
            'name' => 'multiple',
            'label' => __('quizzes.crud_fields.multiple'),
            'type' => 'checkbox',
            'attributes' => [
                'id' => 'free_answer'
            ]
        ]);

        $this->crud->addField([
            'name' => 'is_private',
            'label' => __('quizzes.crud_fields.is_private'),
            'type' => 'checkbox',
            'attributes' => [
                'id' => 'is_private'
            ]
        ]);

        $this->crud->orderBy('created_at', 'DESC');
        $this->crud->removeButton('delete');
        $this->crud->removeButton('update');
        $this->crud->removeButton('preview');
        $this->crud->addButtonFromView('line', 'results', 'results', 'beginning');
        $this->crud->addButton('line', 'show', 'view', 'vendor.backpack.crud.quiz.buttons.preview', 'end');
        $this->crud->allowAccess(['list', 'create', 'show']);
    }

    /**
     * @param UpdateRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(CrudRequest $request)
    {
        $this->crud->hasAccessOrFail('create');

        $request->request->add(['object_id' => null]);

        // Валидатор для основного контекста
        $main_validator = Validator::make($request->all(), [
            'title' => 'required|max:100',
            'question' => 'required|max:1000',
            'start_at' => 'required',
            'end_at' => 'required',
            'destinations' => 'required_if:is_all_house,0',
            'answers' => 'required_if:free_answer,0'
        ]);

        $errors = [];
        $main_validator_fails = $main_validator->fails();
        if ($main_validator_fails) {
            $errors = array_merge($errors, $main_validator->errors()->toArray());
        }

        // Подготовка к валидации таблицы вопросов
        $answers = json_decode($request->input('answers'));
        $variants_json = $answers? $answers : [];

        $variants_rules = [];
        $variants = [];
        foreach ($variants_json as $key => $variant) {
            if (isset($variant->text)) {
                $variants[$key] = $variant->text;
            } else {
                $variants[$key] = '';
            }
            $variants_rules[$key] = 'required|string';
        }

        // Валидация таблицы вопросов
        $variants_validator = Validator::make($variants, $variants_rules);

        $variants_validator_fails = $variants_validator->fails();
        if ($variants_validator_fails) {
            $errors['variants'] = [trans('quizzes.errors.variants')];
        }

        // Если хотябы одна из валидаций не прошла, возвращается обратно с ошибкой
        if ($main_validator_fails || $variants_validator_fails) {
            return redirect()->back()->withErrors($errors)->withInput();
        }

        // fallback to global request instance
        if (is_null($request)) {
            $request = \Request::instance();
        }

        if ($request->get('is_all_house') == 1) {
            $request->merge(['destinations' => \GuzzleHttp\json_encode([['is_all_house' => TRUE]])]);

        }

        $response = parent::storeCrud();

        $destinations = \GuzzleHttp\json_decode($request->get('destinations'), TRUE);

        $apartments = new Apartment();
        if (!empty($apartmentsNumbers)) {
            $apartments->whereIn('number', $apartmentsNumbers);
        }

        foreach ($destinations as $destination) {
            $destination['quiz_id'] = $this->data['entry']->id;
            (new QuizDestination)->create($destination);
        }

        $this->updateOrCreateVariants(json_decode($request->input('answers')));
        $response->setTargetUrl($request->getUri());

        return $response;

    }

    /**
     * @param UpdateRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(UpdateRequest $request)
    {
        $issetModelsAnswers = (new QuizAnswer())->where('quiz_id', $request->input('id'))->get()->toArray();
        $response = parent::updateCrud();
        $this->updateOrCreateVariants(json_decode($request->input('answers')),$issetModelsAnswers);
        return \Redirect::to($this->crud->route);
    }


    public function show($id) {
        $this->crud->hasAccessOrFail('show');
        $this->data['entry'] = $this->crud->getEntry($id);
        $this->data['crud'] = $this->crud;
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['title'] = trans('backpack::crud.preview') . ' ' . $this->crud->entity_name;
        unset($this->data['fields']['template']);
        foreach ($this->data['fields'] as $key => &$field) {
            $field['attributes']['readonly'] = true;
            $field['attributes']['disabled'] = true;
        }
        $this->data['id'] = $id;
        return view($this->crud->getShowView(), $this->data);
    }


    /**
     * @return array|\Illuminate\Contracts\View\Factory|\Illuminate\View\View|mixed
     */
    public function create() {
        $this->crud->hasAccessOrFail('create');

        // prepare the fields you need to show
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getCreateFields();
        $this->data['title'] = trans('backpack::crud.add') . ' ' . $this->crud->entity_name;
        $this->setTemplatesData();

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getCreateView(), $this->data);
    }

    /**
     * @return array|\Illuminate\Contracts\View\Factory|\Illuminate\View\View|mixed
     */
    public function edit($id) {
        $this->crud->hasAccessOrFail('update');

        // get the info for that entry
        $this->data['entry'] = $this->crud->getEntry($id);
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['title'] = trans('backpack::crud.edit') . ' ' . $this->crud->entity_name;

        $this->data['id'] = $id;

        $this->setTemplatesData();

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getEditView(), $this->data);
    }

    /**
     * @param $id
     * @return array|\Illuminate\Contracts\View\Factory|\Illuminate\View\View|mixed
     */
    public function results($id){
        if(Quiz::find($id)->object_id != config('app.object.id')){
            return \Redirect::to($this->crud->route);
        }
        $this->prepareResultsData($id);
        return view('crud::quiz.results',
            array_merge($this->data,['printUrl' => '/' . $this->crud->route . '/' . $this->crud->entry->id . '/results/print'])
            );
    }

    /**
     * @param $id
     * @return array|\Illuminate\Contracts\View\Factory|\Illuminate\View\View|mixed
     */
    public function printResults($id) {
        $this->prepareResultsData($id);
        return view('crud::quiz.print',
            array_merge($this->data,['backUrl' => '/' . $this->crud->route . '/' . $this->crud->entry->id . '/results'])
        );
    }

    /**
     * Update quiz answers variants or create new
     *
     * @param $issetAnswers
     * @param $newAnswers
     */
    private function updateOrCreateVariants($newAnswers, $issetAnswers = []) {

        // in request answers can be null, but parameter is always sent to method, so we can't use default value
        $newAnswers = $newAnswers ?: [];

        $arIds = [];
        $variant = new QuizAnswer;
        foreach($newAnswers as $answer) {
            if($answer->text){
                if(!isset($answer->id)) {
                    $variant->create(['quiz_id' => $this->crud->entry->id, 'text' => $answer->text]);
                    $arIds[] = $variant->id;
                } else {
                    $variant = $variant->find($answer->id);
                    $variant->text = $answer->text;
                    $variant->save();
                    $arIds[] = $variant->id;
                }
            }
        }
        if($issetAnswers) {
            foreach ($issetAnswers as $issetAnswer) {
                if (!in_array($issetAnswer['id'], $arIds)) {
                    $variant->find($issetAnswer['id'])->delete();
                }
            }
        }
    }

    /**
     * Prepare quiz results data for views
     * @param $id
     */
    private function prepareResultsData($id) {
        $this->data['entry'] = $this->crud->entry = Quiz::withCount('results')->find($id);
        $this->data['answers'] = QuizAnswer::with('results')->where('quiz_id' , $id)->withCount('results')->get();
        if($this->crud->entry->free_answer && $this->crud->entry->results_count) {
            $count = QuizResult::where('quiz_id', $id)
                ->where('free_answer_value', '!=', null)
                ->count();
            $free_answers = QuizResult::where('quiz_id', $id)
                ->where('free_answer_value', '!=', null)->get();
            $free_answers = $this->handleFreeAnswers($free_answers, $id);
            $this->data['free_answers'] = $free_answers;
            $this->data['free_answers_percent'] = round($count * 100 / $this->crud->entry->results_count,2);
        } else {
            $this->data['free_answers_percent'] = 0;
        }
        $this->data['empty'] = false;
        if(!$this->crud->entry->results_count){
            $this->data['empty'] = true;
        }
        $this->data['crud'] = $this->crud;
        $this->data['title'] = trans('quizzes.results').': '.$this->crud->entry->title;
        $this->data['status'] = $this->crud->entry->status;
    }

    private function handleFreeAnswers($free_answers, $id) {
        $final_freeanswers = [];
        $quiz_destinations = QuizDestination::where('quiz_id', $id)->get();
        $quiz_is_all_house = $this->isQuizForAllHouse($quiz_destinations);
        foreach ($free_answers as $free_answer) {
            $apartments_of_user = User::find($free_answer->user_id)->apartments()->where('object_id', Cookie::get('object'))->pluck('number');

            if ($quiz_is_all_house && count($apartments_of_user) == 1) {
                $apartment_number = $apartments_of_user[0];
            } else {
                $apartment_number = implode(';', $this->getDestinationApartments($apartments_of_user, $quiz_destinations));
            }
            $value = $free_answer->free_answer_value;
            if (array_key_exists($value, $final_freeanswers)) {
                if ($final_freeanswers[$value] != "") {
                    $final_freeanswers[$value] = $final_freeanswers[$value].', №'.$apartment_number;
                } else {
                    $final_freeanswers[$value] .= '№'.$apartment_number;
                }
            } else {
                $final_freeanswers[$value] = '№'.$apartment_number;
            }
        }
        return $final_freeanswers;
    }

    private function getDestinationApartments($apartments, $destinations) {
        $destination_apartments = [];
        foreach ($apartments as $number) {
            foreach ($destinations as $destination) {
                if ($number == $destination->apartment) {
                    $destination_apartments[] = $number;
                    continue;
                }
                $userApartment = Apartment::where('number',$number)->where('object_id', Cookie::get('object'))->get();
                $user_apartment = $userApartment[0];
                if ($user_apartment->entrance == $destination->entrance) {
                    $destination_apartments[] = $number;
                    continue;
                }
                if ($user_apartment->floor == $destination->floor) {
                    $destination_apartments[] = $number;
                    continue;
                }
            }
        }

        return $destination_apartments;
    }

    private function isQuizForAllHouse($destinations) {
        if ($destinations[0]->is_all_house == 1) {
            return true;
        }
        return false;
    }

    protected function setTemplatesData() {
        $this->data['templatePickerTitle'] = trans('quizzes.add_template');
        $this->data['templateStyle'] = Auth::user()->getStyleType();

        switch ($this->data['templateStyle']) {
            case 'list':
                $this->data['templates'] = QuizTemplate::with('answers')->get();

                break;
            case 'tags':
                $templateTags = QuizTemplate::where('tags','!=', null)->get()->pluck('tags')->toArray();
                $tags = [];
                foreach ($templateTags as $tplTag){
                    $tags = array_merge(explode(',',$tplTag), $tags);
                }
                $this->data['tags'] = array_values(array_unique($tags));
                $this->data['templates'] = QuizTemplate::with('answers')->get();
                break;

            case 'tree':
                $this->data['categories'] = collect(Category::all())
                    ->sortBy('lft')
                    ->keyBy((new Category)->getKeyName());
                foreach ($this->data['categories'] as $key => $entry) {
                    $entry->items = $entry->quizTemplates()->get();
                }
                break;
        }
    }

    // Перегрузка метода для поиска и выделения жирным активных записей
    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // overwrite any order set in the setup() method with the datatables order
        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);
            if ($column['tableColumn']) {
                // clear any past orderBy rules
                $this->crud->query->getQuery()->orders = null;
                // apply the current orderBy rules
                if(in_array($column['name'], ['title'])) {
                    // для поля с именем number установлена отдельная сортировка для более удобочитаемого отображения квартир, имеющих номера
                    $this->crud->query = $this->crud->query->orderByRaw(
                        $column['name'] . '+0 '.$column_direction.', '.
                        $column['name'].' '.$column_direction.
                        ', LENGTH('.$column['name'].') '.$column_direction
                    );
                } else {
                    $this->crud->orderBy($column['name'], $column_direction);
                }
            }
        }
        $entries = $this->crud->getEntries();

        $data = $this->crud->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);

        /* Сформировать массив идентификаторов, которые будут выделяться жирным
         * inProgress - которые будут выделяться жирным
         * isNotPublished - которые будут выделяться очень жирным
         */
        $entries = $this->crud->getEntries();

        $inProgress = collect();
        $isNotPublished = collect();

        for ($i = 0; $i < count($entries); $i++) {
            if ($entries[$i]->getStatus() === 0) {
                $inProgress->push($i);
            } elseif ($entries[$i]->getStatus() === 2) {
                $isNotPublished->push($i);
            }
        }

        foreach ($inProgress as $progressId) {
            $data['data'][$progressId]['DT_RowClass'] = 'font-weight-900';
        }

        foreach ($isNotPublished as $notPublishedId) {
            $data['data'][$notPublishedId]['DT_RowClass'] = 'font-weight-600';
        }

        return $data;
    }
}
