<?php

namespace App\Http\Controllers\Admin;

use App\Models\Category;
use App\Models\MessageTemplate;
use App\Models\PermissionManager\Permission;
use App\Traits\Admin\Ordering;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\CRUD\app\Http\Controllers\CrudController;

// VALIDATION: change the requests to match your own file names if you need form validation
use App\Http\Requests\MessageTemplateRequest as StoreRequest;
use App\Http\Requests\MessageTemplateRequest as UpdateRequest;

class MessageTemplateCrudController extends CrudController {

    use CheckingPermissions, Ordering {
        createBuilder as createBuilderParent;
    }

    /**
     * MessageTemplateCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->applyCheckingPermissions([[
            [Permission::MESSAGE_TEMPLATES_ACCESS],
            [
                'create',
                'store',
                'index',
                'edit',
                'update',
                'destroy',
            ],
        ]]);
    }

    public function setup() {

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\MessageTemplate');
        $this->crud->setRoute(config('backpack.base.route_prefix') . '/templates/messages');
        $this->crud->setEntityNameStrings(__('templates.messages.singular'), __('templates.messages.plural'));

        $this->applyBackpackPermissions();

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */

        // ------ CRUD FIELDS

        $this->crud->addField([
            'name' => 'theme',
            'label' => __('templates.messages.crud_fields.theme'),
        ]);

        $this->crud->addField([
            'name' => 'text',
            'label' => __('templates.messages.crud_fields.text'),
            'type' => 'textarea',
        ]);

        $this->crud->addField([
            'name' => 'category',
            'value' => '<i class="fa fa-folder-open" aria-hidden="true"></i>&nbsp;&nbsp;
                <span class="chosen-category">' . __('templates.categories.category_empty') . '</span>&nbsp;&nbsp;
                <button class="btn btn-primary" type="button" data-toggle="modal" data-target="#categories-picker">'
                . __('templates.categories.buttons.view') . '</button>',
            'type' => 'custom_html',
        ]);

        $this->crud->addField([
            'name' => 'category_id',
            'type' => 'hidden',
            'attributes' => [
                'id' => 'field-category_id',
            ],
        ]);

        // ------ CRUD COLUMNS
        $this->crud->addColumn([
            'name' => 'text',
            'label' => __('templates.messages.crud_fields.text'),
        ]);

        $this->crud->addColumn([
            'name' => 'theme',
            'label' => __('templates.messages.crud_fields.theme'),
        ]);

        $this->crud->addColumn([
            'label' => __('templates.messages.crud_fields.category_id'),
            'type' => 'select',
            'name' => 'category_id',
            'entity' => 'category',
            'attribute' => 'name',
            'model' => "App\Models\Category",
        ]);

        $templates = MessageTemplate::where('tags', '!=', NULL)
            ->get()
            ->pluck('tags')
            ->toArray();
        $tags = [];
        foreach ($templates as $template) {
            $tags = array_merge(explode(',', $template), $tags);
        }
        $tags = array_values(array_unique($tags));

        $this->crud->addField([
            'name' => 'tags',
            'label' => __('templates.quiz.crud_fields.tags'),
            'type' => 'tags',
            'variants' => \GuzzleHttp\json_encode($tags),
        ]);

        // ------ CRUD BUTTONS

        // ------ CRUD ACCESS

        // ------ CRUD REORDER

        // ------ CRUD DETAILS ROW

        // ------ REVISIONS

        // ------ AJAX TABLE VIEW

        // ------ DATATABLE EXPORT BUTTONS

        // ------ ADVANCED QUERIES

        $this->crud->setCreateView('crud::templates.message.create');
        //        $this->crud->setCreateView('crud::adverts.create');
    }

    public function store(StoreRequest $request) {
        // your additional operations before save here
        $redirect_location = parent::storeCrud($request);
        // your additional operations after save here
        // use $this->data['entry'] or $this->crud->entry
        return $redirect_location;
    }

    public function update(UpdateRequest $request) {
        // your additional operations before save here
        $redirect_location = parent::updateCrud($request);
        // your additional operations after save here
        // use $this->data['entry'] or $this->crud->entry
        return $redirect_location;
    }

    /**
     * Show the form for creating inserting a new row.
     *
     * @return Response
     */
    public function create() {
        $this->crud->hasAccessOrFail('create');

        // prepare the fields you need to show
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getCreateFields();
        $this->data['title'] = trans('backpack::crud.add') . ' ' . $this->crud->entity_name;
        $this->data['categories'] = collect(Category::all())
            ->sortBy('lft')
            ->keyBy((new Category)->getKeyName());

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getCreateView(), $this->data);
    }

    public function show($id){
        Abort(404);
    }

    /**
     * Переопределение необходимо для сортировки ненастраиваемых полей
     *
     * @return array|\Backpack\CRUD\app\Http\Controllers\Operations\JSON
     * @throws \Backpack\CRUD\Exception\AccessDeniedException
     */
    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // Кастомная сортировка
        $sortedEntriesIds = null;
        $sortedEntries = collect();
        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);

            $customOrder = true;
            $table = $this->crud->getModel()->getTable();
            switch ($column['name']) {
                // По столбцу текст
                case 'text':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'text', [
                        'object' => false
                    ]);
                    break;
                // По столбцу тема
                case 'theme':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'theme', [
                        'object' => false
                    ]);
                    break;
                // По столбцу категория
                case 'category_id':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'name', [
                        'object' => false
                    ]);
                    break;
                default:
                    $customOrder = false;
                    if ($column['tableColumn']) {
                        // clear any past orderBy rules
                        $this->crud->query->getQuery()->orders = null;
                        // apply the current orderBy rules
                        $this->crud->orderBy($column['name'], $column_direction);
                    }
                    break;
            }

            // Если сортировка кастомная, то сформировать коллекцию объектов
            if ($customOrder) {
                $sortedEntries = $this->createOrderedObjectsByIds('App\Models\MessageTemplate', $sortedEntriesIds);
            }
        }

        // Если сортировка кастомная, то присваиваем отсортированные записи
        $entries = $sortedEntries->count()? $sortedEntries : $this->crud->getEntries();

        return $this->crud->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);
    }

    /**
     * Переопределение построителя для подключения таблицы категорий
     *
     * @param $table
     * @param null $selectRaw
     * @param bool $object
     * @param null $clause
     * @return \Illuminate\Database\Query\Builder
     */
    protected function createBuilder($table, $selectRaw = null, $object = true, $clause = null)
    {
        $raw_builder = $this->createBuilderParent($table, $selectRaw, $object, $clause);

        $raw_builder
            ->join('categories', 'categories.id', '=', $this->crud->getModel()->getTable(). '.category_id')
            ->select($this->crud->getModel()->getTable() . '.*', 'categories.name');

        return $raw_builder;
    }
}
