<?php

namespace App\Http\Controllers\Admin;

use App\Models\PermissionManager\Permission;
use App\Traits\Admin\Ordering;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\CRUD\app\Http\Controllers\CrudController;
// VALIDATION: change the requests to match your own file names if you need form validation
use App\Http\Requests\CategoryRequest as StoreRequest;
use App\Http\Requests\CategoryRequest as UpdateRequest;

class CategoryCrudController extends CrudController
{
    use CheckingPermissions, Ordering {
        createBuilder as createBuilderParent;
    }

    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::QUIZ_TEMPLATES_ACCESS, Permission::MESSAGE_TEMPLATES_ACCESS],
            [
                'create',
                'store',
                'index',
                'edit',
                'update',
                'destroy',
                'reorder',
            ],
        ]]);

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel("App\Models\Category");
        $this->crud->setRoute(config('backpack.base.route_prefix', 'admin').'/templates/categories');
        $this->crud->setEntityNameStrings(__('categories.category_singular'), __('categories.category_plural'));

        /*
        |--------------------------------------------------------------------------
        | COLUMNS AND FIELDS
        |--------------------------------------------------------------------------
        */

        $this->crud->allowAccess('reorder');
        $this->crud->enableReorder('name', 2);

        // ------ CRUD COLUMNS
        $this->crud->addColumn([
            'name' => 'name',
            'label' => __('categories.crud_fields.name'),
        ]);

        $this->crud->addColumn([
            'label' => __('categories.crud_fields.parent'),
            'type' => 'select',
            'name' => 'parent_id',
            'entity' => 'parent',
            'attribute' => 'name',
            'model' => "App\Models\Category",
        ]);

        // ------ CRUD FIELDS
        $this->crud->addField([
            'name' => 'name',
            'label' => __('categories.crud_fields.name'),
        ]);

        $this->crud->addField([
            'label' => __('categories.crud_fields.parent'),
            'type' => 'select',
            'name' => 'parent_id',
            'entity' => 'parent',
            'attribute' => 'name',
            'model' => "App\Models\Category",
        ]);
    }

    public function setup()
    {
        $this->applyBackpackPermissions();
    }

    public function store(StoreRequest $request)
    {
        return parent::storeCrud();
    }

    public function update(UpdateRequest $request)
    {
        return parent::updateCrud();
    }

    public function show($id){
        Abort(404);
    }


    /**
     * Переопределение необходимо для сортировки ненастраиваемых полей
     *
     * @return array|\Backpack\CRUD\app\Http\Controllers\Operations\JSON
     * @throws \Backpack\CRUD\Exception\AccessDeniedException
     */
    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // Кастомная сортировка
        $sortedEntriesIds = null;
        $sortedEntries = collect();
        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);

            $customOrder = true;
            $table = $this->crud->getModel()->getTable();
            switch ($column['name']) {
                // По столбцу текст
                case 'name':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'categories.name', [
                        'object' => false
                    ]);
                    break;
                // По столбцу родительская категория
                case 'parent_id':
                    $sortedEntriesIds = $this->sortBySimpleRow($table, 'parent', [
                        'object' => false
                    ]);
                    break;
                default:
                    $customOrder = false;
                    if ($column['tableColumn']) {
                        // clear any past orderBy rules
                        $this->crud->query->getQuery()->orders = null;
                        // apply the current orderBy rules
                        $this->crud->orderBy($column['name'], $column_direction);
                    }
                    break;
            }

            // Если сортировка кастомная, то сформировать коллекцию объектов
            if ($customOrder) {
                $sortedEntries = $this->createOrderedObjectsByIds('App\Models\Category', $sortedEntriesIds);
            }
        }

        // Если сортировка кастомная, то присваиваем отсортированные записи
        $entries = $sortedEntries->count()? $sortedEntries : $this->crud->getEntries();

        return $this->crud->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);
    }

    protected function createBuilder($table, $selectRaw = null, $object = true, $clause = null)
    {
        $builder = $this->createBuilderParent($table, $selectRaw, $object, $clause);

        $builder
            ->join('categories as parent_categories', 'categories.parent_id', '=', 'parent_categories.id', 'left')
            ->select('categories.*', 'parent_categories.name AS parent');

        return $builder;
    }
}