<?php

namespace App\Http\Controllers\Admin;

use App\Models\PermissionManager\Permission;
use App\Traits\Admin\Ordering;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\CRUD\app\Http\Controllers\CrudController;

// VALIDATION: change the requests to match your own file names if you need form validation
use App\Http\Requests\CameraRequest as StoreRequest;
use App\Http\Requests\CameraRequest as UpdateRequest;
use Illuminate\Support\Facades\Auth;

class CameraCrudController extends CrudController
{
    use CheckingPermissions, Ordering;

    /**
     * @var string[] $displayed_fields_names
     */
    public static $displayed_fields_names = [
        'name',
        'rtsp_link',
    ];

    /**
     * CameraCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::CAMERAS_CREATE_ACCESS],
            ['create', 'store'],
        ], [
            [Permission::CAMERAS_EDIT_ACCESS, Permission::CAMERAS_VIEW_LIMITED_ACCESS],
            ['index'],
        ], [
            [Permission::CAMERAS_EDIT_ACCESS],
            ['edit', 'update'],
        ], [
            [Permission::CAMERAS_DELETE_ACCESS],
            ['destroy']
        ]]);
    }

    public function setup()
    {
        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\Camera');
        $this->crud->setRoute(config('backpack.base.route_prefix') . '/camera');
        $this->crud->setEntityNameStrings(__('cameras.camera_singular'), __('cameras.camera_plural'));

        $this->setButtons();
        $this->applyBackpackPermissions();

        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        foreach (self::$displayed_fields_names as $displayed_field_name) {
            $translated_field_name = __("cameras.crud_fields.$displayed_field_name");

            // Add field to create/edit form
            $this->crud->addField([
                'name' => $displayed_field_name,
                'label' => $translated_field_name,
                'type' => 'text',
            ]);

            if (
                ($displayed_field_name == 'name')
                ||
                Auth::user()->can(Permission::CAMERAS_EDIT_ACCESS)
            ) {
                $this->crud->addColumn([
                    'name' => $displayed_field_name,
                    'label' => $translated_field_name,
                    'type' => 'text',
                ]);
            }
        }

        $this->crud->addField([
            'name' => 'object_id',
            'type' => 'hidden',
            'value' => config('app.object.id')
        ]);

        $this->crud->addField([
            'label' => __('apartments.crud_fields.binding_to_apartments'),
            'type' => 'apartments_checklist',
            'name' => 'apartments',
            'entity' => 'apartments',
            'attributes' => [
                'number' => __('apartments.apartment')
            ],
            'object_id' => config('app.object.id'),
            'model' => "\App\Models\Apartment",
            'pivot' => true,
            'check_all' => false,
        ], 'update');

        $this->crud->addField([
            'label' => __('apartments.crud_fields.binding_to_apartments'),
            'type' => 'apartments_checklist',
            'name' => 'apartments',
            'entity' => 'apartments',
            'attributes' => [
                'number' => __('apartments.apartment')
            ],
            'object_id' => config('app.object.id'),
            'model' => "\App\Models\Apartment",
            'pivot' => true,
            'check_all' => true,
        ], 'create');

        $this->crud->addClause('where', 'object_id', '=', config('app.object.id'));

        // ------ CRUD FIELDS
        // $this->crud->addField($options, 'update/create/both');
        // $this->crud->addFields($array_of_arrays, 'update/create/both');
        // $this->crud->removeField('name', 'update/create/both');
        // $this->crud->removeFields($array_of_names, 'update/create/both');


        // ------ CRUD COLUMNS
        // $this->crud->addColumn(); // add a single column, at the end of the stack
        // $this->crud->addColumns(); // add multiple columns, at the end of the stack
        // $this->crud->removeColumn('column_name'); // remove a column from the stack
        // $this->crud->removeColumns(['column_name_1', 'column_name_2']); // remove an array of columns from the stack
        // $this->crud->setColumnDetails('column_name', ['attribute' => 'value']); // adjusts the properties of the passed in column (by name)
        // $this->crud->setColumnsDetails(['column_1', 'column_2'], ['attribute' => 'value']);


        // ------ CRUD BUTTONS
        // possible positions: 'beginning' and 'end'; defaults to 'beginning' for the 'line' stack, 'end' for the others;
        // $this->crud->addButton($stack, $name, $type, $content, $position); // add a button; possible types are: view, model_function
        // $this->crud->addButtonFromModelFunction($stack, $name, $model_function_name, $position); // add a button whose HTML is returned by a method in the CRUD model
        // $this->crud->addButtonFromView($stack, $name, $view, $position); // add a button whose HTML is in a view placed at resources\views\vendor\backpack\crud\buttons
        // $this->crud->removeButton($name);
        // $this->crud->removeButtonFromStack($name, $stack);
        // $this->crud->removeAllButtons();
        // $this->crud->removeAllButtonsFromStack('line');

        // ------ CRUD ACCESS
        // $this->crud->allowAccess(['list', 'create', 'update', 'reorder', 'delete']);
        // $this->crud->denyAccess(['list', 'create', 'update', 'reorder', 'delete']);

        // ------ CRUD REORDER
        // $this->crud->enableReorder('label_name', MAX_TREE_LEVEL);
        // NOTE: you also need to do allow access to the right users: $this->crud->allowAccess('reorder');

        // ------ CRUD DETAILS ROW
        // $this->crud->enableDetailsRow();
        // NOTE: you also need to do allow access to the right users: $this->crud->allowAccess('details_row');
        // NOTE: you also need to do overwrite the showDetailsRow($id) method in your EntityCrudController to show whatever you'd like in the details row OR overwrite the views/backpack/crud/details_row.blade.php

        // ------ REVISIONS
        // You also need to use \Venturecraft\Revisionable\RevisionableTrait;
        // Please check out: https://laravel-backpack.readme.io/docs/crud#revisions
        // $this->crud->allowAccess('revisions');

        // ------ AJAX TABLE VIEW
        // Please note the drawbacks of this though:
        // - 1-n and n-n columns are not searchable
        // - date and datetime columns won't be sortable anymore
        // $this->crud->enableAjaxTable();

        // ------ DATATABLE EXPORT BUTTONS
        // Show export to PDF, CSV, XLS and Print buttons on the table view.
        // Does not work well with AJAX datatables.
        // $this->crud->enableExportButtons();

        // ------ ADVANCED QUERIES
        // $this->crud->addClause('active');
        // $this->crud->addClause('type', 'car');
        // $this->crud->addClause('where', 'name', '==', 'car');
        // $this->crud->addClause('whereName', 'car');
        // $this->crud->addClause('whereHas', 'posts', function($query) {
        //     $query->activePosts();
        // });
        // $this->crud->addClause('withoutGlobalScopes');
        // $this->crud->addClause('withoutGlobalScope', VisibleScope::class);
        // $this->crud->with(); // eager load relationships
        // $this->crud->orderBy();
        // $this->crud->groupBy();
        // $this->crud->limit();

        $this->crud->setCreateView('crud::cameras.create');
        $this->crud->setEditView('crud::cameras.edit');
    }

    public function store(StoreRequest $request)
    {
        // your additional operations before save here
        $redirect_location = parent::storeCrud($request);
        // your additional operations after save here
        // use $this->data['entry'] or $this->crud->entry
        return $redirect_location;
    }

    public function update(UpdateRequest $request)
    {
        // your additional operations before save here
        $redirect_location = parent::updateCrud($request);
        // your additional operations after save here
        // use $this->data['entry'] or $this->crud->entry
        return $redirect_location;
    }

    public function edit($id)
    {
        $this->crud->hasAccessOrFail('update');

        // get the info for that entry
        $this->data['entry'] = $this->crud->getEntry($id);
        // get the info for that entry
        $this->data['entry'] = $this->crud->getEntry($id);
        if($this->data['entry']->object_id  != config('app.object.id')){
            return \Redirect::to($this->crud->route);
        }
        $this->data['crud'] = $this->crud;
        $this->data['saveAction'] = $this->getSaveAction();
        $this->data['fields'] = $this->crud->getUpdateFields($id);
        $this->data['title'] = trans('backpack::crud.edit').' '.$this->crud->entity_name;

        $this->data['id'] = $id;

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getEditView(), $this->data);
    }

    protected function setButtons()
    {
        $this->crud->removeAllButtons();

        if (Auth::user()->can(Permission::CAMERAS_CREATE_ACCESS)) {
            $this->crud->addButton('top', 'create', 'view', 'crud::buttons.create');
        }
        if (Auth::user()->can(Permission::CAMERAS_DELETE_ACCESS)) {
            $this->crud->addButton('line', 'delete', 'view', 'crud::buttons.delete');
        }
        if (Auth::user()->can(Permission::CAMERAS_EDIT_ACCESS)) {
            $this->crud->addButton('line', 'update', 'view', 'crud::buttons.update');
        }
    }

    /**
     * Get the save configured save action or the one stored in a session variable.
     * @return [type] [description]
     */
    public function getSaveAction()
    {
        $saveAction = session('save_action', config('backpack.crud.default_save_action', 'save_and_back'));
        $saveOptions = [];
        $saveCurrent = [
            'value' => $saveAction,
            'label' => $this->getSaveActionButtonName($saveAction),
        ];

        switch ($saveAction) {
            case 'save_and_edit':
                $saveOptions['save_and_back'] = $this->getSaveActionButtonName('save_and_back');
                if (Auth::user()->can(Permission::CAMERAS_CREATE_ACCESS)) {
                    $saveOptions['save_and_new'] = $this->getSaveActionButtonName('save_and_new');
                }
                break;
            case 'save_and_new':
                if (Auth::user()->can(Permission::CAMERAS_CREATE_ACCESS)) {
                    $saveOptions['save_and_back'] = $this->getSaveActionButtonName('save_and_back');
                }
                if (Auth::user()->can(Permission::CAMERAS_EDIT_ACCESS) || Auth::user()->can(Permission::USER_EDIT_FULL_ACCESS)) {
                    $saveOptions['save_and_edit'] = $this->getSaveActionButtonName('save_and_edit');
                }
                break;
            case 'save_and_back':
            default:
                if (Auth::user()->can(Permission::CAMERAS_EDIT_ACCESS) || Auth::user()->can(Permission::USER_EDIT_FULL_ACCESS)) {
                    $saveOptions['save_and_edit'] = $this->getSaveActionButtonName('save_and_edit');
                }
                if (Auth::user()->can(Permission::CAMERAS_CREATE_ACCESS)) {
                    $saveOptions['save_and_new'] = $this->getSaveActionButtonName('save_and_new');
                }
                break;
        }

        return [
            'active' => $saveCurrent,
            'options' => $saveOptions,
        ];
    }

    /**
     * Get the translated text for the Save button.
     * @param  string $actionValue [description]
     * @return [type]              [description]
     */
    private function getSaveActionButtonName($actionValue = 'save_and_back')
    {
        switch ($actionValue) {
            case 'save_and_edit':
                return trans('backpack::crud.save_action_save_and_edit');
                break;
            case 'save_and_new':
                return trans('backpack::crud.save_action_save_and_new');
                break;
            case 'save_and_back':
            default:
                return trans('backpack::crud.save_action_save_and_back');
                break;
        }
    }

    public function show($id){
        Abort(404);
    }

    public function search()
    {
        $this->crud->hasAccessOrFail('list');

        $totalRows = $filteredRows = $this->crud->count();
        $startIndex = $this->request->input('start') ?: 0;
        // if a search term was present
        if ($this->request->input('search') && $this->request->input('search')['value']) {
            // filter the results accordingly
            $this->crud->applySearchTerm($this->request->input('search')['value']);
            // recalculate the number of filtered rows
            $filteredRows = $this->crud->count();
        }
        // start the results according to the datatables pagination
        if ($this->request->input('start')) {
            $this->crud->skip($this->request->input('start'));
        }
        // limit the number of results according to the datatables pagination
        if ($this->request->input('length')) {
            $this->crud->take($this->request->input('length'));
        }
        // overwrite any order set in the setup() method with the datatables order
        $rawSortedEntries = null;
        $sortedEntries = collect();
        if ($this->request->input('order')) {
            $column_number = $this->request->input('order')[0]['column'];
            $column_direction = $this->request->input('order')[0]['dir'];
            $column = $this->crud->findColumnById($column_number);

            $customOrder = true;
            $table = $this->crud->getModel()->getTable();
            switch ($column['name']) {
                // По столбцу название
                case 'name':
                    $rawSortedEntries = $this->sortBySimpleRow($table, 'name');
                    break;
                // По столбцу ссылка
                case 'rtsp_link':
                    $rawSortedEntries = $this->sortBySimpleRow($table, 'rtsp_link', [
                        'type' => 'length'
                    ]);
                    break;
                default:
                    $customOrder = false;
                    if ($column['tableColumn']) {
                        // clear any past orderBy rules
                        $this->crud->query->getQuery()->orders = null;
                        // apply the current orderBy rules
                        $this->crud->orderBy($column['name'], $column_direction);
                    }
                    break;
            }

            // Если сортировка кастомная, то сформировать коллекцию объектов
            if ($customOrder) {
                $sortedEntries = $this->createOrderedObjectsByIds('App\Models\Camera', $rawSortedEntries);
            }
        }

        // Если сортировка кастомная, то присваиваем отсортированные записи
        $entries = $sortedEntries->count()? $sortedEntries : $this->crud->getEntries();

        return $this->crud->getEntriesAsJsonForDatatables($entries, $totalRows, $filteredRows, $startIndex);
    }

    public function camerasCreate() {

    }
}
