<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use App\Models\PermissionManager\Permission;
use App\Traits\PermissionManager\CheckingPermissions;
use Backpack\CRUD\app\Http\Requests\CrudRequest;
use Backpack\PermissionManager\app\Http\Controllers\UserCrudController as BackpackUserCrudController;
use Backpack\PermissionManager\app\Http\Requests\UserStoreCrudRequest as StoreRequest;
use Backpack\PermissionManager\app\Http\Requests\UserUpdateCrudRequest as UpdateRequest;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\UserPermissionsWeight;
use Illuminate\Support\Facades\DB;

/**
 * Class UserCrudController
 *
 * @package App\Http\Controllers\Admin
 */
class ActiveUsersController extends BackpackUserCrudController
{
    use CheckingPermissions;

    /**
     * @var string $permissionsRelation
     */
    protected $permissionsRelation = 'permissions';

    /**
     * @var string $rolesRelation
     */
    protected $rolesRelation = 'roles';


    /**
     * UserCrudController constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->applyCheckingPermissions([[
            [Permission::FULL_ACCESS, Permission::AUTHORIZED_USERS_FULL_ACCESS, Permission::AUTHORIZED_USERS_LIMITED_ACCESS],
            ['index']
        ]]);
    }


    /**
     * Allow developers to set their configuration options for a CrudPanel.
     */
    public function setup()
    {
        $this->applyBackpackPermissions();
        /*
        |--------------------------------------------------------------------------
        | BASIC CRUD INFORMATION
        |--------------------------------------------------------------------------
        */
        $this->crud->setModel('App\Models\User');
        $this->crud->setEntityNameStrings(__('active_users.singular'), __('active_users.plural'));
        $this->crud->setRoute(config('backpack.base.route_prefix').'/active-users');

        $user_active_duration = env('USER_ACTIVE_DURATION', 2);
        $max_session_life_time = ini_get("session.gc_maxlifetime");

        $remember_me_users = User::where('remember_marker', 1)->pluck('id')->toArray();
        $session_users = User::where('last_activity', '>=', Carbon::now()->subSeconds($max_session_life_time))
            ->where('remember_marker', 0)
            ->where('manual_exit', 0)
            ->pluck('id')->toArray();

        $users = collect(array_merge($remember_me_users, $session_users));

        if (!Auth::user()->hasPermissionTo(Permission::AUTHORIZED_USERS_FULL_ACCESS)) {
            $users_copy = clone($users);
            $users = collect();

            DB::table('permission_users')
                ->where('object_id', '=', config('app.object.id'))
                ->whereIn('user_id', $users_copy)
                ->join('permissions', 'permissions.id', '=', 'permission_users.permission_id')
                ->selectRaw('user_id, SUM(weight) AS weight')
                ->groupBy('user_id')
                ->get()
                ->each(function ($user) use ($users) {
                    if ($user->weight <= Auth::user()->getPermissionsWeight()) {
                        $users->push($user->user_id);
                    }
            });
        }

        $this->crud->addClause('whereIn', 'id', $users);

        $this->crud->enableAjaxTable();

        // Columns.
        $this->crud->setColumns([
            [
                'name'  => 'name',
                'label' => trans('active_users.crud.name'),
                'type'  => 'text',
            ],
            [
                'name'  => 'email',
                'label' => trans('active_users.crud.email'),
                'type'  => 'email',
            ],[
                'name'  => 'last_activity',
                'label' => trans('active_users.crud.last_activity'),
                'type'  => 'datetime',
            ],
        ]);

        $this->crud->addColumn([
            'name' => "permissions_weight",
            'label' => trans('active_users.crud.permissions_weight'),
            'type' => "model_function",
            'function_name' => 'getPermissionsWeight',
        ]);

        $this->crud->removeAllButtons();
    }


    /**
     * Display all rows in the database for this entity.
     *
     * @return Response
     */
    public function index()
    {
        $this->crud->hasAccessOrFail('list');
        $this->data['crud'] = $this->crud;

        $this->data['title'] = __('active_users.crud.nav_string');

        // get all entries if AJAX is not enabled
        if (!$this->data['crud']->ajaxTable()) {
            $this->data['entries'] = $this->data['crud']->getEntries();
        }

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view($this->crud->getListView(), $this->data);
    }

    public function show($id){
        Abort(404);
    }
}
